/**
 * kuaike.com Inc. Copyright (c) 2014-2020 All Rights Reserved.
 */
package com.kuaike.skynet.logic.wechat.utils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.google.common.collect.Lists;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import lombok.Data;
import lombok.ToString;

/**
 * 对根据现有的关键词建立索引，用于从文本中提取指定的关键词(敏感词)。
 * 
 * <pre>
 * 例: 中国,中国人民,中国话
 * 中 = {
 *   end = 0
 *   国 = {
 *     end = 1 
 *     人 = {
 *       end = 0 
 *       民 = {
 *         end = 1
 *       }
 *     }
 *     话 = {
 *       end = 1 
 *     }
 *   }
 * }
 * </pre>
 * 
 * @title WordTree
 * @author yanmaoyuan
 * @date 2020年4月7日
 * @version 1.0
 * @see <a href="https://blog.csdn.net/chenssy/article/details/26961957">Java实现敏感词过滤</a>
 */
@Data
@ToString
public class WordTree implements Serializable {

    private static final long serialVersionUID = 1L;

    public final static int MIN_MATCH_TYPE = 1; // 最小匹配规则

    public final static int MAX_MATCH_TYPE = 2; // 最大匹配规则

    /**
     * 用于标识DFA中某节点是否为敏感词的终止节点，用于确认匹配到了关键词。
     */
    private boolean isEnd = false;

    /**
     * 用于记录当前节点指向的后续节点。
     */
    private HashMap<Character, WordTree> next = new HashMap<>();

    // for serialize only
    public WordTree() {
    }

    /**
     * 根据一批关键词建立索引。
     * 
     * @param words
     */
    public WordTree(Collection<String> words) {
        for (String word : words) {
            addWord(word);
        }
    }

    /**
     * 批量添加关键词
     * 
     * @param words
     */
    public void addWords(Collection<String> words) {
        for (String word : words) {
            addWord(word);
        }
    }

    /**
     * 添加关键词，建立索引。
     * 
     * @param word
     */
    public void addWord(String word) {
        if (word == null || word.isEmpty()) {
            return;
        }

        WordTree cur = this;// rootNode

        int len = word.length();
        for (int i = 0; i < len; i++) {
            char keyChar = word.charAt(i); // 获取当前字符

            WordTree next = cur.next.get(keyChar); // 获取

            if (next == null) {
                next = new WordTree();
                cur.next.put(keyChar, next);
            }
            cur = next;

            if (i == len - 1) {
                cur.isEnd = true;
            }
        }
    }

    @JsonIgnore
    public boolean isEmpty() {
        return next.isEmpty();
    }

    /**
     * 查找关键词
     * 
     * @param txt 需要检测的文本
     * @param beginIndex 检测起始索引
     * @param matchType 检测方式
     * @return 如果存在则返回关键词的长度，不存在返回0
     */
    public int checkWord(String txt, int beginIndex, int matchType) {
        boolean flag = false; // 敏感词结束标识位

        WordTree cur = this;// rootNode
        
        int length = txt.length();
        int curLength = 0; // 当前匹配的敏感词长度

        for (int i = beginIndex; i < length; i++) {
            char word = txt.charAt(i);
            cur = cur.next.get(word); // 获取指定key

            // 没有找到字符对应的索引，终止查找。
            if (cur == null) {
                break;
            }
            curLength++;

            if (cur.isEnd) {// 如果为最后一个匹配规则,结束循环，返回匹配标识数
                flag = true; // 结束标志位为true
                if (MIN_MATCH_TYPE == matchType) { // 最小规则，直接返回,最大规则还需继续查找
                    break;
                }
            }
        }

        if (!flag) {
            curLength = 0;
        }

        return curLength;
    }

    /**
     * 判断文字是否包含敏感字符
     * 
     * @param txt 文字
     * @return 若包含返回true，否则返回false
     * @version 1.0
     */
    public boolean isContaintWord(String txt) {
        if (txt == null || txt.isEmpty()) {
            return false;
        }

        if (this.next.isEmpty()) {
            return false;
        }

        boolean flag = false;
        for (int i = 0; i < txt.length(); i++) {
            int matchFlag = checkWord(txt, i, MIN_MATCH_TYPE); // 判断是否包含敏感字符
            if (matchFlag > 0) { // 大于0存在，返回true
                flag = true;
            }
        }
        return flag;
    }

    /**
     * 获取文字中的敏感词
     * 
     * @param txt 文字
     * @return
     * @version 1.0
     */
    public List<String> getMatchedWords(String txt) {
        Set<String> matched = new LinkedHashSet<>();
        if (txt == null || txt.isEmpty()) {
            return Collections.emptyList();
        }

        if (this.next.isEmpty()) {
            return Collections.emptyList();
        }

        int length = txt.length();
        for (int i = 0; i < length; i++) {

            WordTree cur = this;

            int curLength = 0; // 匹配标识数默认为0
            int beginIndex = i;
            for (int k = beginIndex; k < length; k++) {
                char word = txt.charAt(k);
                cur = cur.next.get(word); // 获取指定key

                if (cur == null) {// 不存在，直接返回
                    break;
                }

                curLength++; // 找到相应key，匹配标识+1
                if (cur.isEnd) {
                    // 存在,加入list中
                    String curWord = txt.substring(beginIndex, beginIndex + curLength);
                    matched.add(curWord);
                }
            }
        }

        return Lists.newArrayList(matched);
    }

    public final static void main(String[] args) {
        Set<String> words = new HashSet<>();
        words.add("爱国");
        words.add("和谐");
        words.add("富强");

        WordTree tree = new WordTree();
        tree.addWords(words);
        System.out.println(tree);

        System.out.println("敏感词的数量：" + tree.next.size());
        StringBuffer sb = new StringBuffer();
        sb.append("社会主义核心价值观的基本内容是：");
        sb.append("富强、民主、文明、和谐、自由、平等、公正、法治、爱国、敬业、诚信、友善。");
        sb.append("“富强、民主、文明、和谐”，是我国社会主义现代化国家的建设目标；");
        sb.append("“自由、平等、公正、法治”，是对美好社会的生动表述；");
        sb.append("“爱国、敬业、诚信、友善”，是公民基本道德规范，是从个人行为层面对社会主义核心价值观基本理念的凝练。");
        String string = sb.toString();
        System.out.println("待检测语句字数：" + string.length());
        List<String> set = null;

        long beginTime = System.currentTimeMillis();
        int n = 400000;
        for (int i = 0; i < n; i++) {
            set = tree.getMatchedWords(string);
        }

        long endTime = System.currentTimeMillis();
        System.out.println("语句中包含敏感词的个数为：" + set.size() + "。包含：" + set);
        System.out.println("循环检测" + n + "次总共消耗时间为：" + (endTime - beginTime) + "ms");
    }
}