package com.kuaike.skynet.logic.wechat.utils;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.google.common.collect.Lists;
import com.kuaike.skynet.logic.wechat.dto.ReferMsgDto;
import com.kuaike.skynet.logic.wechat.dto.ReferMsgRespDto;
import com.kuaike.skynet.logic.wechat.enums.ReferMsgType;
import com.kuaike.skynet.logic.wechat.msg.LocationMsg;
import com.kuaike.skynet.logic.wechat.msg.QuoteMsg;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.LinkedList;
import java.util.Objects;

@Slf4j
public class ReferMsgResolver {

    private static final XmlMapper xmlMapper;

    static {
        xmlMapper = new XmlMapper();
        xmlMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    }

    public static ReferMsgRespDto parseReferMsg(String message) {
        ReferMsgRespDto dto = new ReferMsgRespDto();
        try {
            message = getInvalidMsg(message);
            QuoteMsg quoteMsg = xmlMapper.readValue(message, QuoteMsg.class);
            QuoteMsg.QuoteAppMsg appmsg = quoteMsg.getAppmsg();
            dto.setContent(appmsg.getTitle());
            LinkedList<ReferMsgDto> referMsgDtoList = Lists.newLinkedList();
            collectReferMsg(message, referMsgDtoList);
            dto.setRefer(referMsgDtoList);
        } catch (Exception e) {
            log.error("解析引用消息失败:{}", message, e);
        }

        return dto;
    }

    private static ReferMsgRespDto parse(String title, QuoteMsg.ReferMsg referMsg) {
        String quote = "";

        StringBuilder builder = new StringBuilder();
        String displayName = referMsg.getDisplayName();
        builder.append(displayName).append(": ");

        String text = referMsg.getContent();
        log.info("解析引用消息：{}", text);
        Integer type = referMsg.getType();
        ReferMsgType referMsgType = ReferMsgType.getByReferType(type);
        if (Objects.isNull(referMsgType)) {
            return null;
        }
        switch (referMsgType) {
            case TEXT:
                builder.append(text);
                break;
            case IMAGE:
            case VIDEO:
            case GIF:
                builder.append("[").append(referMsgType.getDesc()).append("]");
                break;
            case LOCATION:
                try {
                    text = getInvalidMsg(text);
                    //引用位置类型
                    LocationMsg locationMsg = xmlMapper.readValue(text, LocationMsg.class);
                    if (Objects.nonNull(locationMsg.getLocation()) && StringUtils.isNotBlank(locationMsg.getLocation().getPoiName())) {
                        quote = locationMsg.getLocation().getPoiName();
                    }
                } catch (Exception e) {
                    log.warn("忽略具体引用内容:{}", text);
                }
                builder.append("[").append(referMsgType.getDesc()).append("]").append(quote);
                break;
            case LINK:
            case File:
            case MINI_PROGRAM:
            case REFER:
                ReferMsgType msgType = null;
                try {
                    text = getInvalidMsg(text);
                    //引用文件 小程序等类型
                    QuoteMsg appMsg = xmlMapper.readValue(text, QuoteMsg.class);
                    QuoteMsg.QuoteAppMsg appmsg = appMsg.getAppmsg();
                    if (Objects.nonNull(appmsg) && Objects.nonNull(appmsg.getType()) && StringUtils.isNotBlank(appmsg.getTitle())) {
                        quote = appmsg.getTitle();
                        msgType = ReferMsgType.getByType(appmsg.getType());
                    }
                } catch (Exception e) {
                    log.warn("忽略具体引用内容:{}", text);
                }
                if (Objects.isNull(msgType)) {
                    return null;
                }
                if (msgType == ReferMsgType.REFER) {
                    builder.append(quote);
                } else {
                    builder.append("[").append(msgType.getDesc()).append("]").append(quote);
                }
                break;
            default:
                return null;
        }

        ReferMsgRespDto dto = new ReferMsgRespDto();
        dto.setContent(title);
        //dto.setRefer(builder.toString());
        return dto;
    }


    private static String getInvalidMsg(String msg) throws Exception {
        String start = "<msg";
        String end = "</msg>";
        int index = msg.indexOf(start);
        int endIndex = msg.indexOf(end);
        if (index == -1 || endIndex == -1) {
            throw new Exception("消息内容格式错误，无法解析");
        }
        msg = msg.substring(index, endIndex + end.length());
        return msg;
    }

    public static void collectReferMsg(String message, LinkedList<ReferMsgDto> referMsgSrvIdList) {
        try {
            QuoteMsg quoteMsg = xmlMapper.readValue(message, QuoteMsg.class);
            if (Objects.nonNull(quoteMsg)) {
                QuoteMsg.QuoteAppMsg appmsg = quoteMsg.getAppmsg();
                if (Objects.nonNull(appmsg)) {
                    QuoteMsg.ReferMsg referMsg = appmsg.getReferMsg();
                    if (Objects.nonNull(referMsg)) {
                        String svrid = referMsg.getSvrid();
                        String displayName = referMsg.getDisplayName();
                        ReferMsgDto dto = new ReferMsgDto();
                        dto.setNickname(displayName);
                        dto.setMsgSrvId(svrid);
                        dto.setType(referMsg.getType());
                        //文本消息，先从引用里面取消息内容，避免根据msgSrvId查询不到
                        if(referMsg.getType() == 1) {
                            dto.setContent(referMsg.getContent());
                        }
                        referMsgSrvIdList.addFirst(dto);
                        collectReferMsg(referMsg.getContent(), referMsgSrvIdList);
                    }
                }
            }
        } catch (Exception e) {
            log.debug("解析引用消息失败:{}", message, e);
        }
    }

}
