package com.kuaike.skynet.logic.wechat.utils;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.google.common.collect.Lists;
import com.kuaike.common.entity.WechatMessage;
import com.kuaike.skynet.logic.wechat.msg.JoinGroupMsg;
import com.kuaike.skynet.logic.wechat.msg.JoinGroupMsg.Link;
import com.kuaike.skynet.logic.wechat.msg.JoinGroupMsg.Member;

import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.List;

import lombok.extern.slf4j.Slf4j;

/**
 * 入群消息解析工厂
 * 
 * @title JoinGroupFactory
 * @author yanmaoyuan
 * @date 2020年9月16日
 * @version 1.0
 */
@Slf4j
public final class JoinGroupFactory {

    /**
     * 用于解析xml文件
     */
    public final static XmlMapper xmlMapper;

    static {
        xmlMapper = new XmlMapper();
        xmlMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    }

    private JoinGroupFactory() {}

    interface Parser {
        JoinGroupDto parse(Member me, JoinGroupMsg msg);
    }

    public static JoinGroupDto parse(WechatMessage wechatMessage) {
        Member me = new Member();
        me.setUsername(wechatMessage.getWechatId());
        me.setNickname(wechatMessage.getNickName());

        // 获取xml
        String message = wechatMessage.getMessage();
        String start = "<sysmsg";
        String end = "</sysmsg>";
        int startIdx = message.indexOf(start);
        int endIdx = message.indexOf(end);
        String xml;
        if (startIdx < 0 || endIdx < 0) {
            log.warn("消息内容格式错误，无法解析, requestId={}", wechatMessage.getRequestId());
            return null;
        } else {
            xml = message.substring(startIdx, endIdx + end.length());
        }

        // 解析入群消息的xml
        JoinGroupMsg msg = null;
        try {
            msg = xmlMapper.readValue(xml, JoinGroupMsg.class);
        } catch (IOException e) {
            e.printStackTrace();
            log.error("消息解析失败, requestId:{}", wechatMessage.getRequestId(), e);
            return null;
        }

        // 根据消息模版，获取对应的解析器
        Parser parser = getParser(msg.getSysMsgTemplate().getContentTemplate().getTemplate());
        if (parser == null) {
            log.info("Not join group msg, template={}", msg.getSysMsgTemplate().getContentTemplate().getTemplate());
            return null;
        }

        // 解析入群消息内容
        try {
            return parser.parse(me, msg);
        } catch (Exception e) {
            e.printStackTrace();
            log.error("Parse join group message failed, xml={}", xml, e);
            return null;
        }
    }

    /**
     * 解析器工厂，根据模板内容返回对应的解析器。
     * 
     * @param template
     * @return
     */
    private static Parser getParser(String template) {
        if (StringUtils.isBlank(template)) {
            return null;
        }

        switch (template) {
            // 某人邀请了你和其他人加入群聊
            case "\"$username$\"邀请你和\"$names$\"加入了群聊":
            case "「$username$」邀請你和$names$加入了群組":
            case "invited you and $names$ to the group chat":
                return invitedYouAndOthers();
            // 某人邀请了其他人加入群聊
            case "\"$username$\"邀请\"$names$\"加入了群聊":
            case "「$username$」邀請「$names$」加入了群組":
            case "$username$ invited $names$ to the group chat":
                return invitedOthers();
            // 你邀请了其他人加入群聊
            case "你邀请\"$names$\"加入了群聊  $revoke$":
            case "你邀請「$names$」加入了群組  $revoke$":
            case "You invited $names$ to the group chat.   $revoke$":
                return youInvitedOthers();
            // 某人邀请你加入了群聊
            case "\"$username$\"邀请你加入了群聊，群聊参与人还有：$others$":
            case "「$username$」邀請你加入了群組，成員還有：$others$":
            case "$username$ invited you to a group chat with $others$":
                return invitedYou();
            // 某人邀请你加入了群聊
            case "\"$username$\"邀请你加入了群聊":
            case "「$username$」邀請你加入了群組":
            case "$username$ invited you to the group chat":
                return invitedYou();
            // 扫描你的二维码进群
            case "\"$adder$\"通过扫描你分享的二维码加入群聊  $revoke$":
            case "「$adder$」透過掃描你分享的QR code加入群組  $revoke$":
            case "\"$adder$\" joined group chat via the QR code you shared.  $revoke$":
                return scanYourQrcode();
            // 扫描其他人的二维码进群
            case "\" $adder$\"通过扫描\"$from$\"分享的二维码加入群聊":
            case "「$adder$」透過掃描「$from$」分享的QR Code加入群組。":
            case "\"$adder$\" joined the group chat via the QR Code shared by \"$from$\".":
                return scanQrcode();
            // 面对面建群
            case "\"$names$\"加入了群聊":
            case "「$names$」加入了群組":
            case "\"$names$\" joined the group chat.":
                return create();
            default:
                return null;
        }
    }

    private static Parser invitedYouAndOthers() {
        return new Parser() {
            public JoinGroupDto parse(Member me, JoinGroupMsg msg) {
                List<Link> linklist = msg.getSysMsgTemplate().getContentTemplate().getLinklist();
                Member inviteMember = null;
                List<Member> memberList = null;

                for (Link link : linklist) {
                    if ("username".equals(link.getName())) {
                        inviteMember = link.getMemberList().get(0);
                    } else if ("names".equals(link.getName())) {
                        memberList = link.getMemberList();
                        memberList.add(me);
                    }
                }

                return new JoinGroupDto(JoinGroupDto.INVITE, inviteMember, memberList);
            }
        };
    }

    private static Parser invitedOthers() {
        return new Parser() {
            public JoinGroupDto parse(Member me, JoinGroupMsg msg) {
                List<Link> linklist = msg.getSysMsgTemplate().getContentTemplate().getLinklist();
                Member inviter = null;
                List<Member> memberList = null;

                for (Link link : linklist) {
                    if ("username".equals(link.getName())) {
                        inviter = link.getMemberList().get(0);
                    } else if ("names".equals(link.getName())) {
                        memberList = link.getMemberList();
                    }
                }

                return new JoinGroupDto(JoinGroupDto.INVITE, inviter, memberList);
            }
        };
    }

    private static Parser youInvitedOthers() {
        return new Parser() {
            public JoinGroupDto parse(Member me, JoinGroupMsg msg) {
                List<Link> linklist = msg.getSysMsgTemplate().getContentTemplate().getLinklist();
                Member inviter = me;
                List<Member> memberList = null;

                for (Link link : linklist) {
                    if ("names".equals(link.getName())) {
                        memberList = link.getMemberList();
                    }
                }

                return new JoinGroupDto(JoinGroupDto.INVITE, inviter, memberList);
            }
        };
    }

    private static Parser invitedYou() {
        return new Parser() {
            public JoinGroupDto parse(Member me, JoinGroupMsg msg) {
                List<Link> linklist = msg.getSysMsgTemplate().getContentTemplate().getLinklist();
                Member inviter = null;
                List<Member> memberList = Lists.newArrayList(me);

                for (Link link : linklist) {
                    if ("username".equals(link.getName())) {
                        inviter = link.getMemberList().get(0);
                    }
                }

                return new JoinGroupDto(JoinGroupDto.INVITE, inviter, memberList);
            }
        };
    }

    private static Parser scanYourQrcode() {
        return new Parser() {
            public JoinGroupDto parse(Member me, JoinGroupMsg msg) {
                List<Link> linklist = msg.getSysMsgTemplate().getContentTemplate().getLinklist();
                Member inviter = me;
                List<Member> memberList = null;

                for (Link link : linklist) {
                    if ("adder".equals(link.getName())) {
                        memberList = link.getMemberList();
                    }
                }

                return new JoinGroupDto(JoinGroupDto.SCAN_CODE, inviter, memberList);
            }
        };
    }

    private static Parser scanQrcode() {
        return new Parser() {
            public JoinGroupDto parse(Member me, JoinGroupMsg msg) {
                List<Link> linklist = msg.getSysMsgTemplate().getContentTemplate().getLinklist();
                Member inviter = null;
                List<Member> memberList = null;

                for (Link link : linklist) {
                    if ("adder".equals(link.getName())) {
                        memberList = link.getMemberList();
                    } else if ("from".equals(link.getName())) {
                        inviter = link.getMemberList().get(0);
                    }
                }

                return new JoinGroupDto(JoinGroupDto.SCAN_CODE, inviter, memberList);
            }
        };
    }

    private static Parser create() {
        return new Parser() {
            public JoinGroupDto parse(Member me, JoinGroupMsg msg) {
                List<Link> linklist = msg.getSysMsgTemplate().getContentTemplate().getLinklist();
                Member inviter = me;
                List<Member> memberList = null;

                for (Link link : linklist) {
                    if ("names".equals(link.getName())) {
                        memberList = link.getMemberList();
                    }
                }

                return new JoinGroupDto(JoinGroupDto.CREATE, inviter, memberList);
            }
        };
    }
}