package cn.kinyun.scrm.weixin.sdk.api.shop;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.shop.req.*;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.AfterSaleBatchGetResp;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.AfterSaleGetResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Objects;

/**
 * @author zhangjing
 * @date 2022/1/5 9:25 上午
 */
@Slf4j
@Component
public class WxShopAfterSaleAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 售后接口-获取售后单 POST
     */
    @Value("${wx.shop.after_sale.get_after_sale_order}")
    private String shopGetAfterSaleOrderUrl;

    /**
     * 售后接口-批量获取售后单 POST
     */
    @Value("${wx.shop.after_sale.batch_get_after_sale_order}")
    private String shopBatchGetAfterSaleOrderUrl;

    /**
     * 售后接口-同意售后 POST
     */
    @Value("${wx.shop.after_sale.accept_apply}")
    private String shopAfterSaleAcceptApplyUrl;

    /**
     * 售后接口-拒绝售后 POST
     */
    @Value("${wx.shop.after_sale.reject_refund}")
    private String shopAfterSaleRejectRefundUrl;

    /**
     * 售后接口-调用地址 POST
     */
    @Value("${wx.shop.after_sale.upload_refund_certificate}")
    private String shopAfterSaleUploadRefundCertificateUrl;

    /**
     * 获取售后单
     *
     * @param accessToken
     * @param req
     * @return
     */
    public AfterSaleGetResp getAfterSaleOrder(@NonNull String accessToken, @NonNull AfterSaleGetReq req) {
        log.info("getAfterSaleOrder with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<AfterSaleGetResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<AfterSaleGetReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopGetAfterSaleOrderUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, AfterSaleGetResp.class);

        AfterSaleGetResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 批量获取售后单
     *
     * @param accessToken
     * @param req
     * @return
     */
    public AfterSaleBatchGetResp batchGetAfterSaleOrder(@NonNull String accessToken, @NonNull AfterSaleBatchGetReq req) {
        log.info("batchGetAfterSaleOrder with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<AfterSaleBatchGetResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<AfterSaleBatchGetReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopBatchGetAfterSaleOrderUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, AfterSaleBatchGetResp.class);

        AfterSaleBatchGetResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 同意售后
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode acceptApply(@NonNull String accessToken, @NonNull AfterSaleAcceptApplyReq req) {
        log.info("acceptApply with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<AfterSaleAcceptApplyReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopAfterSaleAcceptApplyUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 拒绝售后
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode rejectRefund(@NonNull String accessToken, @NonNull AfterSaleRejectRefundReq req) {
        log.info("rejectRefund with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<AfterSaleRejectRefundReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopAfterSaleRejectRefundUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 上传退款凭证
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode uploadRefundCertificate(@NonNull String accessToken, @NonNull AfterSaleUploadRefundCertificateReq req) {
        log.info("uploadRefundCertificate with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<AfterSaleUploadRefundCertificateReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopAfterSaleUploadRefundCertificateUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
