/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api.analysis;

import cn.kinyun.scrm.weixin.sdk.entity.analysis.AnalysisReq;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.msg.UpstreamMsg;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.msg.UpstreamMsgDist;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.msg.UpstreamMsgHour;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 微信消息数据分析接口
 * 
 * <p>
 * 在公众号登录授权机制的权限集划分中，消息分析数据接口属于消息管理权限。
 * </p>
 * 
 * @see <a href=
 *      "https://developers.weixin.qq.com/doc/offiaccount/Analytics/Message_analysis_data_interface.html">数据统计-消息分析</a>
 * @title WxMsgAnalysisAPI
 * @author yanmaoyuan
 * @date 2019年9月24日
 * @version 1.0
 */
@Slf4j
@Component
public class WxMsgAnalysisAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 获取消息发送概况数据 POST
     */
    @Value("${wx.analysis.getupstreammsg}")
    private String wxAnalysisGetUpstreamMsg;

    /**
     * 获取消息发送分时数据 POST
     */
    @Value("${wx.analysis.getupstreammsghour}")
    private String wxAnalysisGetUpstreamMsgHour;

    /**
     * 获取消息发送周数据 POST
     */
    @Value("${wx.analysis.getupstreammsgweek}")
    private String wxAnalysisGetUpstreamMsgWeek;

    /**
     * 获取消息发送月数据 POST
     */
    @Value("${wx.analysis.getupstreammsgmonth}")
    private String wxAnalysisGetUpstreamMsgMonth;

    /**
     * 获取消息发送分布数据 POST
     */
    @Value("${wx.analysis.getupstreammsgdist}")
    private String wxAnalysisGetUpstreamMsgDist;

    /**
     * 获取消息发送分布周数据 POST
     */
    @Value("${wx.analysis.getupstreammsgdistweek}")
    private String wxAnalysisGetUpstreamMsgDistWeek;

    /**
     * 获取消息发送分布月数据 POST
     */
    @Value("${wx.analysis.getupstreammsgdistmonth}")
    private String wxAnalysisGetUpstreamMsgDistMonth;

    /**
     * 获取消息发送概况数据
     * 
     * <p>
     * 最大时间跨度:7天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UpstreamMsg getUpstreamMsg(@NonNull String accessToken, @NonNull AnalysisReq params) throws WeixinException {
        log.info("get upstream msg with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUpstreamMsg, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UpstreamMsg> response = restTemplate.postForEntity(url, request, UpstreamMsg.class);

        UpstreamMsg result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取消息发送分时数据
     * 
     * <p>
     * 最大时间跨度:1天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UpstreamMsgHour getUpstreamMsgHour(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get upstream msg hour with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUpstreamMsgHour, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UpstreamMsgHour> response = restTemplate.postForEntity(url, request, UpstreamMsgHour.class);

        UpstreamMsgHour result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取消息发送周数据
     * 
     * <p>
     * 最大时间跨度:30天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UpstreamMsg getUpstreamMsgWeek(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get upstream msg week with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUpstreamMsgWeek, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UpstreamMsg> response = restTemplate.postForEntity(url, request, UpstreamMsg.class);

        UpstreamMsg result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取消息发送月数据
     * 
     * <p>
     * 最大时间跨度:30天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UpstreamMsg getUpstreamMsgMonth(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get upstream msg month with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUpstreamMsgMonth, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UpstreamMsg> response = restTemplate.postForEntity(url, request, UpstreamMsg.class);

        UpstreamMsg result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取消息发送分布数据
     * 
     * <p>
     * 最大时间跨度:15天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UpstreamMsgDist getUpstreamMsgDist(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get upstream msg dist with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUpstreamMsgDist, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UpstreamMsgDist> response = restTemplate.postForEntity(url, request, UpstreamMsgDist.class);

        UpstreamMsgDist result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取消息发送分布周数据
     * 
     * <p>
     * 最大时间跨度:30天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UpstreamMsgDist getUpstreamMsgDistWeek(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get upstream msg dist week with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUpstreamMsgDistWeek, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UpstreamMsgDist> response = restTemplate.postForEntity(url, request, UpstreamMsgDist.class);

        UpstreamMsgDist result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取消息发送分布月数据
     * 
     * <p>
     * 最大时间跨度:30天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UpstreamMsgDist getUpstreamMsgDistMonth(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get upstream msg dist month with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUpstreamMsgDistMonth, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UpstreamMsgDist> response = restTemplate.postForEntity(url, request, UpstreamMsgDist.class);

        UpstreamMsgDist result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}