/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.enums;

import java.util.HashMap;
import java.util.Map;

/**
 * 微信消息类型
 * 
 * @title WxMsgType
 * @desc 微信消息类型
 * @author yanmaoyuan
 * @date 2019年4月24日
 * @version 1.0
 */
@SuppressWarnings("squid:S115")
public enum WxMsgType {

    /**
     * 1 文本消息
     */
    Text("text"),

    /**
     * 2 图片消息
     */
    Image("image"),

    /**
     * 3 语音消息
     */
    Voice("voice"),

    /**
     * 4 视频消息
     */
    Video("video"),

    /**
     * 5 小视频消息
     */
    ShortVideo("shortvideo"),

    /**
     * 6 地理位置消息
     */
    Location("location"),

    /**
     * 7 链接消息
     */
    Link("link"),

    /**
     * 事件消息
     * 
     * @see WxEventType
     */
    Event("event"),

    /**
     * 音乐消息
     */
    Music("music"),

    /**
     * 图文消息
     */
    News("news"),

    /**
     * 上传至微信的图文消息素材
     */
    MpNews("mpnews"),

    /**
     * 上传至微信的群发视频素材
     */
    MpVideo("mpvideo"),

    /**
     * 微信卡劵
     */
    WxCard("wxcard"),

    /**
     * 客服给用户发送的菜单消息
     * 
     * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140547">客服接口-发消息</a>
     * @see com.kuaike.weixin.api.WxCustomMsgApi#checkCustomMsgType
     */
    MsgMenu("msgmenu"),

    /**
     * 客服给用户发送的小程序卡片
     * 
     * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140547">客服接口-发消息</a>
     * @see com.kuaike.weixin.api.WxCustomMsgApi#checkCustomMsgType
     */
    MiniProgramPage("miniprogrampage"),

    /**
     * 将消息转发到客服
     * 
     * <p>
     * 如果公众号处于开发模式，普通微信用户向公众号发消息时，微信服务器会先将消息POST到开发者填写的url上，如果希望将消息转发到客服系统，则需要开发者在响应包中返回MsgType为transfer_customer_service的消息，微信服务器收到响应后会把当次发送的消息转发至客服系统。
     * </p>
     * 
     * <p>
     * 您也可以在返回transfer_customer_service消息时，在XML中附上TransInfo信息指定分配给某个客服帐号。
     * </p>
     * 
     * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1458557405">将消息转发到客服</a>
     */
    TransferCustomerService("transfer_customer_service"),

    ;

    private static final Map<String, WxMsgType> cache = new HashMap<>();

    static {
        for (WxMsgType e : values()) {
            cache.put(e.getValue(), e);
        }
    }

    public static WxMsgType get(String msgType) {
        return cache.get(msgType);
    }

    private String value;

    private WxMsgType(String value) {
        this.value = value;
    }

    public String getValue() {
        return value;
    }

    public static WxMsgType fromMaterialType(String materialType) {
        switch (materialType) {
            case "news":
                return WxMsgType.MpNews;
            case "video":
                return WxMsgType.Video;
            case "text":
                return WxMsgType.Text;
            case "image":
                return WxMsgType.Image;
            case "voice":
                return WxMsgType.Voice;
            default:
                return null;
        }
    }
}