package cn.kinyun.scrm.weixin.sdk.api.shop;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.shop.req.*;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.*;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Objects;

/**
 * @author zhangjing
 * @date 2022/1/5 4:04 下午
 */
@Slf4j
@Component
public class WxShopCouponAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 优惠券接口-创建优惠券 POST
     */
    @Value("${wx.shop.coupon.create}")
    private String shopCouponCreateUrl;

    /**
     * 优惠券接口-更新优惠券 POST
     */
    @Value("${wx.shop.coupon.update}")
    private String shopCouponUpdateUrl;

    /**
     * 优惠券接口-更新优惠券状态 POST
     */
    @Value("${wx.shop.coupon.update_status}")
    private String shopCouponUpdateStatusUrl;

    /**
     * 优惠券接口-获取优惠券信息 POST
     */
    @Value("${wx.shop.coupon.get}")
    private String shopCouponGetUrl;

    /**
     * 优惠券接口-获取优惠券列表 POST
     */
    @Value("${wx.shop.coupon.get_list}")
    private String shopCouponGetListUrl;

    /**
     * 优惠券接口-发放优惠券 POST
     */
    @Value("${wx.shop.coupon.push}")
    private String shopCouponPushUrl;

    /**
     * 优惠券接口-获取用户优惠券列表 POST
     */
    @Value("${wx.shop.coupon.get_user}")
    private String shopCouponGetUserUrl;

    /**
     * 创建优惠券
     *
     * @param accessToken
     * @param req
     * @return
     */
    public CouponCreateResp createCoupon(@NonNull String accessToken, @NonNull CouponCreateReq req) {
        log.info("createCoupon with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<CouponCreateResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<CouponCreateReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopCouponCreateUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, CouponCreateResp.class);

        CouponCreateResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 更新优惠券
     *
     * @param accessToken
     * @param req
     * @return
     */
    public CouponCreateResp updateCoupon(@NonNull String accessToken, @NonNull CouponCreateReq req) {
        log.info("updateCoupon with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<CouponCreateResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<CouponCreateReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopCouponUpdateUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, CouponCreateResp.class);

        CouponCreateResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 更新优惠券状态
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode updateCouponStatus(@NonNull String accessToken, @NonNull CouponUpdateStatusReq req) {
        log.info("updateCouponStatus with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<CouponUpdateStatusReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopCouponUpdateStatusUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取优惠券信息
     *
     * @param accessToken
     * @param req
     * @return
     */
    public CouponGetResp getCoupon(@NonNull String accessToken, @NonNull CouponGetReq req) {
        log.info("getCoupon with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<CouponGetResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<CouponGetReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopCouponGetUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, CouponGetResp.class);

        CouponGetResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取优惠券列表
     *
     * @param accessToken
     * @param req
     * @return
     */
    public CouponGetListResp getCouponList(@NonNull String accessToken, @NonNull CouponGetListReq req) {
        log.info("getCouponList with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<CouponGetListResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<CouponGetListReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopCouponGetListUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, CouponGetListResp.class);

        CouponGetListResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 发放优惠券
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode pushCoupon(@NonNull String accessToken, @NonNull CouponPushReq req) {
        log.info("pushCoupon with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<CouponPushReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopCouponPushUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取用户优惠券列表
     *
     * @param accessToken
     * @param req
     * @return
     */
    public CouponGetUserListResp getCouponUserList(@NonNull String accessToken, @NonNull CouponGetUserListReq req) {
        log.info("getCouponUserList with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<CouponGetUserListResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<CouponGetUserListReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopCouponGetUserUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, CouponGetUserListResp.class);

        CouponGetUserListResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
