package cn.kinyun.scrm.weixin.sdk.api.material;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.material.dto.NewsContent;
import cn.kinyun.scrm.weixin.sdk.entity.material.dto.VideoMaterialDto;
import cn.kinyun.scrm.weixin.sdk.entity.material.req.AddNewsReq;
import cn.kinyun.scrm.weixin.sdk.entity.material.req.AddVideoReq;
import cn.kinyun.scrm.weixin.sdk.entity.material.req.ModifyNewsReq;
import cn.kinyun.scrm.weixin.sdk.entity.media.Attachment;
import cn.kinyun.scrm.weixin.sdk.enums.WxMediaType;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.alibaba.fastjson.JSONException;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Maps;
import cn.kinyun.scrm.weixin.sdk.entity.HttpResponse;
import cn.kinyun.scrm.weixin.sdk.entity.material.resp.AddMaterialResp;
import cn.kinyun.scrm.weixin.sdk.entity.material.resp.AddNewsResp;
import cn.kinyun.scrm.weixin.sdk.entity.material.resp.GetMaterialCountResp;
import cn.kinyun.scrm.weixin.sdk.entity.material.resp.GetMaterialResp;
import cn.kinyun.scrm.weixin.sdk.entity.material.resp.MaterialResp;
import cn.kinyun.scrm.weixin.sdk.utils.HttpClient;
import cn.kinyun.scrm.weixin.sdk.utils.HttpsClient;

import org.apache.commons.io.IOUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.io.File;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.text.MessageFormat;
import java.util.Map;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * @ClassName MaterialApi
 * @Description
 * @Author zhangjing
 * @DATE 2019/4/28 15:55
 * @Version 1.0
 */
@Component
@Slf4j
public class MaterialApi {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 新增永久图文素材 POST
     */
    @Value("${wx,material.news.add}")
    private String addNewsUrl;

    /**
     * 新增其他类型永久素材 POST
     */
    @Value("${wx.material.add}")
    private String addUrl;

    /**
     * 获取永久素材 POST
     */
    @Value("${wx.material.get}")
    private String getMaterialUrl;

    /**
     * 删除永久素材 POST
     */
    @Value("${wx.material.del}")
    private String delMaterialUrl;

    /**
     * 修改永久图文素材 POST
     */
    @Value("${wx.material.update}")
    private String updateMaterialUrl;

    /**
     * 获取永久素材总数 GET
     */
    @Value("${wx.material.count}")
    private String countUrl;

    /**
     * 获取素材列表
     */
    @Value("${wx.material.list}")
    private String listUrl;

    /**
     * 新增永久图文素材
     *
     * @param accessToken
     * @param req
     * @return
     */
    public AddNewsResp addNews(@NonNull String accessToken, @NonNull AddNewsReq req) throws WeixinException {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AddNewsReq> request = new HttpEntity<AddNewsReq>(req, headers);
        ResponseEntity<AddNewsResp> resp =
            restTemplate.postForEntity(addNewsUrl, request, AddNewsResp.class, accessToken);

        AddNewsResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 新增永久视频素材
     *
     * @param accessToken
     * @param file
     * @param req
     * @return
     */
    public AddMaterialResp addVideo(@NonNull String accessToken, @NonNull File file, @NonNull AddVideoReq req)
        throws WeixinException {
        FileSystemResource resource = new FileSystemResource(file);
        MultiValueMap<String, Object> param = new LinkedMultiValueMap<>();
        param.add("media", resource);
        param.add("filename", file.getName());
        param.add("description", req);
        HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<MultiValueMap<String, Object>>(param);
        ResponseEntity<AddMaterialResp> resp = restTemplate.exchange(addUrl, HttpMethod.POST, httpEntity,
            AddMaterialResp.class, accessToken, WxMediaType.Video.getValue());

        AddMaterialResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 新增其他类型永久素材（新增视频请使用上面的方法）
     *
     * @param accessToken
     * @param type
     * @param file
     * @return
     */
    public AddMaterialResp addMaterial(@NonNull String accessToken, @NonNull WxMediaType type, @NonNull File file)
        throws WeixinException {
        FileSystemResource resource = new FileSystemResource(file);
        MultiValueMap<String, Object> param = new LinkedMultiValueMap<>();
        param.add("media", resource);
        param.add("filename", file.getName());
        HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<MultiValueMap<String, Object>>(param);
        ResponseEntity<AddMaterialResp> resp = restTemplate.exchange(addUrl, HttpMethod.POST, httpEntity,
            AddMaterialResp.class, accessToken, type.getValue());

        AddMaterialResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取永久素材
     *
     * @param accessToken
     * @param mediaId
     * @return
     * @throws Exception
     */
    public GetMaterialResp getMaterial(@NonNull String accessToken, @NonNull String mediaId)
        throws GeneralSecurityException, IOException, WeixinException {
        log.info("get material with media_id={}", mediaId);
        String url = MessageFormat.format(getMaterialUrl, accessToken);

        Map<String, String> map = Maps.newHashMap();
        map.put("media_id", mediaId);
        ObjectMapper mapper = new ObjectMapper();
        String json = mapper.writeValueAsString(map);

        HttpsClient client = new HttpsClient();
        HttpResponse response = client.requestHttps(url, json);
        if (response == null) {
            WeixinException.isSuccess(-1);
            return null;
        }

        GetMaterialResp resp = new GetMaterialResp();
        if (response.getContentType() != null && (response.getContentType().contains("text/plain")
            || response.getContentType().contains("application/json"))) {
            // 视频url,图文素材
            String textString = IOUtils.toString(response.getInputStream(), "UTF-8");
            try {
                JSONObject result = JSONObject.parseObject(textString);
                if (result.containsKey("errcode") && result.getIntValue("errcode") != 0) {
                    resp.setErrMsg(result.getString("errmsg"));
                    resp.setErrCode(result.getInteger("errcode"));
                }
                if (result.containsKey("news_item")) {
                    // 图文素材
                    resp.setType(1);
                    NewsContent content = JSONObject.parseObject(textString, NewsContent.class);
                    resp.setNews(content);
                } else if (result.containsKey("down_url")) {
                    // 视频素材
                    resp.setType(2);
                    VideoMaterialDto video = JSONObject.parseObject(textString, VideoMaterialDto.class);

                    // 下载视频素材
                    HttpClient httpClient = new HttpClient();
                    Attachment attachment = httpClient.download(video.getDownUrl());
                    video.setAttachment(attachment);

                    resp.setVideo(video);
                }
            } catch (JSONException e) {
                
                log.error("解析微信永久素材返回的json失败", e);
                resp.setErrCode(-1);
                resp.setErrMsg("解析json失败:" + textString);
            }
        } else {
            // 文件（语音，缩略图）
            resp.setType(3);
            String ds = response.getContentDisposition();
            // 这里返回的是 attachment; filename=xxxxxxx.mp4
            // 也有可能是 form-data; name="aaa"; filename="xxxxxxxx.mp4"
            String fileName = "";
            if (ds.contains("attachment;")) {
                fileName = ds.substring(ds.indexOf("filename=") + 10, ds.length());
            } else if (ds.contains("form-data;")) {
                fileName = ds.substring(ds.indexOf("filename=\"") + 10, ds.length() - 1);
            }

            Attachment attachment = new Attachment();
            attachment.setContentLength(response.getContentLength());
            attachment.setContentType(response.getContentType());
            attachment.setFileName(fileName);
            attachment.setInputStream(response.getInputStream());
            resp.setAttachment(attachment);
        }
        WeixinException.isSuccess(resp);
        return resp;
    }

    /**
     * 删除永久素材
     *
     * @param accessToken 调用接口凭证
     * @param mediaId 要获取的素材的media_id
     * @return
     */
    public void delMaterial(@NonNull String accessToken, @NonNull String mediaId) throws WeixinException {
        Map<String, String> map = Maps.newHashMap();
        map.put("media_id", mediaId);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<Map<String, String>> request = new HttpEntity<>(map, headers);
        ResponseEntity<ErrorCode> responseEntity =
            restTemplate.postForEntity(delMaterialUrl, request, ErrorCode.class, accessToken);

        ErrorCode result = responseEntity.getBody();
        WeixinException.isSuccess(result);
    }

    /**
     * 修改永久素材
     *
     * @param accessToken
     * @param req
     * @return
     */
    public void modifyNews(@NonNull String accessToken, @NonNull ModifyNewsReq req) throws WeixinException {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<ModifyNewsReq> request = new HttpEntity<>(req, headers);
        ResponseEntity<ErrorCode> responseEntity =
            restTemplate.postForEntity(updateMaterialUrl, request, ErrorCode.class, accessToken);

        ErrorCode result = responseEntity.getBody();
        WeixinException.isSuccess(result);
    }

    /**
     * 获取素材总数
     *
     * @param accessToken
     * @return
     */
    public GetMaterialCountResp getMaterialCount(@NonNull String accessToken) throws WeixinException {
        ResponseEntity<GetMaterialCountResp> responseEntity =
            restTemplate.getForEntity(countUrl, GetMaterialCountResp.class, accessToken);
        GetMaterialCountResp result = responseEntity.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取素材列表
     *
     * @param accessToken
     * @param type 材的类型，图片（image）、视频（video）、语音 （voice）、图文（news）{@link WxMediaType}
     * @param offset 从全部素材的该偏移位置开始返回，0表示从第一个素材 返回
     * @param count 返回素材的数量，取值在1到20之间
     * @return
     */
    public MaterialResp materialList(@NonNull String accessToken, @NonNull WxMediaType type, @NonNull Integer offset,
                                     @NonNull Integer count) throws WeixinException {

        Map<String, Object> map = Maps.newHashMap();
        map.put("type", type.getValue());
        map.put("offset", offset >= 0 ? offset : 0);
        map.put("count", count >= 0 ? count : 20);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<Map<String, Object>> request = new HttpEntity<>(map, headers);
        ResponseEntity<MaterialResp> responseEntity =
            restTemplate.postForEntity(listUrl, request, MaterialResp.class, accessToken);

        MaterialResp result = responseEntity.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
