/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api.custom;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.collect.Maps;
import cn.kinyun.scrm.weixin.sdk.entity.custom.Client;
import cn.kinyun.scrm.weixin.sdk.entity.custom.ClientList;
import cn.kinyun.scrm.weixin.sdk.entity.custom.Servant;
import cn.kinyun.scrm.weixin.sdk.entity.custom.WaitCase;
import cn.kinyun.scrm.weixin.sdk.entity.custom.WaitCaseList;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.List;
import java.util.Map;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 微信客服会话控制接口
 * 
 * @title WxCustomSessionAPI
 * @desc 微信客服会话控制接口
 * @author yanmaoyuan
 * @date 2019年5月5日
 * @version 1.0
 * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1458044820">会话控制</a>
 */
@Slf4j
@Component
public class WxCustomSessionAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 创建会话 POST
     */
    @Value("${wx.custom.kfsession.create}")
    private String wxCustomSessionCreate;

    /**
     * 关闭会话 POST
     */
    @Value("${wx.custom.kfsession.close}")
    private String wxCustomSessionClose;

    /**
     * 获取客户会话状态 GET
     */
    @Value("${wx.custom.kfsession.getsession}")
    private String wxCustomSessionGet;

    /**
     * 获取客服会话列表 GET
     */
    @Value("${wx.custom.kfsession.getsessionlist}")
    private String wxCustomSessionGetList;

    /**
     * 获取未接入会话列表 GET
     */
    @Value("${wx.custom.kfsession.getwaitcase}")
    private String wxCustomSessionGetWaitCase;

    /**
     * 创建会话
     * 
     * @param accessToken 接口调用凭证
     * @param kfAccount 完整客服帐号，格式为：帐号前缀@公众号微信号
     * @param openId 粉丝的openid
     * @throws WeixinException
     */
    public void create(@NonNull String accessToken, @NonNull String kfAccount, @NonNull String openId)
        throws WeixinException {
        log.info("create session between kf_account={} and openid={}", kfAccount, openId);
        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        Map<String, Object> params = Maps.newHashMap();
        params.put("kf_account", kfAccount);
        params.put("openid", openId);

        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxCustomSessionCreate, accessToken);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        WeixinException.isSuccess(response.getBody());// 处理错误码
    }

    /**
     * 关闭会话
     * 
     * @param accessToken 接口调用凭证
     * @param kfAccount 完整客服帐号，格式为：帐号前缀@公众号微信号
     * @param openId 粉丝的openid
     * @throws WeixinException
     */
    public void close(@NonNull String accessToken, @NonNull String kfAccount, @NonNull String openId)
        throws WeixinException {
        log.info("close kfsession between kf_account={} and openid={}", kfAccount, openId);
        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        Map<String, Object> params = Maps.newHashMap();
        params.put("kf_account", kfAccount);
        params.put("openid", openId);

        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxCustomSessionClose, accessToken);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        WeixinException.isSuccess(response.getBody());// 处理错误码
    }

    /**
     * 获取客户会话状态
     * 
     * <p>
     * 此接口获取一个客户的会话，如果不存在，则kf_account为空。
     * </p>
     * 
     * @param accessToken 接口调用凭证
     * @param openId 粉丝的openid
     * @return 客服人员
     * @throws WeixinException
     */
    public Servant getSession(@NonNull String accessToken, @NonNull String openId) throws WeixinException {
        log.info("get session with openid={}", openId);
        // 发送请求
        String url = MessageFormat.format(wxCustomSessionGet, accessToken, openId);
        ResponseEntity<Servant> response = restTemplate.getForEntity(url, Servant.class);

        Servant result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码

        return result;
    }

    /**
     * 获取客服会话列表
     * 
     * @param accessToken 接口调用凭证
     * @param kfAccount 完整客服帐号，格式为：帐号前缀@公众号微信号
     * @return 客服会话列表
     * @throws WeixinException
     */
    public List<Client> getSessionList(@NonNull String accessToken, @NonNull String kfAccount) throws WeixinException {
        log.info("get session list with kf_account={}", kfAccount);
        // 发送请求
        String url = MessageFormat.format(wxCustomSessionGetList, accessToken, kfAccount);
        ResponseEntity<ClientList> response = restTemplate.getForEntity(url, ClientList.class);

        ClientList result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码

        return result.getList();
    }

    /**
     * 获取未接入会话列表
     * 
     * @param accessToken 接口调用凭证
     * @return 未接入会话列表
     * @throws WeixinException
     */
    public List<WaitCase> getWaitCase(@NonNull String accessToken) throws WeixinException {
        log.info("get wait case");
        // 发送请求
        String url = MessageFormat.format(wxCustomSessionGetWaitCase, accessToken);
        ResponseEntity<WaitCaseList> response = restTemplate.getForEntity(url, WaitCaseList.class);

        WaitCaseList result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码

        return result.getList();
    }
}