/*
 * 微信公众平台(JAVA) SDK
 *
 * Copyright (c) 2014, Ansitech Network Technology Co.,Ltd All rights reserved.
 * 
 * http://www.weixin4j.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package cn.kinyun.scrm.weixin.sdk.utils;

import cn.kinyun.scrm.weixin.sdk.entity.HttpResponse;
import cn.kinyun.scrm.weixin.sdk.entity.media.Attachment;
import com.alibaba.fastjson.JSONObject;
import lombok.NonNull;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;

import javax.net.ssl.*;
import java.io.*;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.cert.CertificateException;

@SuppressWarnings({"java:S2093", "java:S4423", "java:S125"})
public class HttpsClient {

    private static final int    CONNECTION_TIMEOUT = 30000;
    private static final int    READ_TIMEOUT       = 30000;
    private static final String POST               = "POST";


    /**
     * 获取https请求连接
     *
     * @param url 连接地址
     * @return https连接对象
     * @throws IOException IO异常
     */
    public HttpsURLConnection getHttpsURLConnection(String url) throws IOException {
        URL urlGet = new URL(url);
        // 创建https请求
        HttpsURLConnection httpsUrlConnection = (HttpsURLConnection) urlGet.openConnection();
        return httpsUrlConnection;
    }

    /**
     *
     * @param httpsUrlConnection HttpsURLConnection实例
     * @param method "post"/"get"
     * @param needCert 是否需要证书，true/false
     * @param partnerId 伙伴ID，needCert=true指定证书时，需要传入
     * @param certPath 证书路径，needCert=true指定证书时，需要传入
     * @param certSecret 证书密码，needCert=true指定证书时，需要传入
     * @throws IOException
     * @throws GeneralSecurityException
     */
    public void setHttpsHeader(HttpsURLConnection httpsUrlConnection, String method, boolean needCert,
        String partnerId, String certPath, String certSecret) throws IOException, GeneralSecurityException {
        if (httpsUrlConnection == null) {
            throw new NullPointerException("httpsUrlConnection can not be null");
        }
        // 不需要维修证书，则使用默认证书
        if (needCert) {
            // 指定读取证书格式为PKCS12
            KeyStore keyStore = KeyStore.getInstance("PKCS12");
            // 读取本机存放的PKCS12证书文件
            try(FileInputStream instream = new FileInputStream(new File(certPath))) {
                // 指定PKCS12的密码
                keyStore.load(instream, partnerId.toCharArray());
            }
            KeyManagerFactory kmf = KeyManagerFactory.getInstance("SunX509");
            kmf.init(keyStore, certSecret.toCharArray());
            // 创建管理jks密钥库的x509密钥管理器，用来管理密钥，需要key的密码
            SSLContext sslContext = SSLContext.getInstance("TLSv1");
            // 构造SSL环境，指定SSL版本为3.0，也可以使用TLSv1，但是SSLv3更加常用。
            sslContext.init(kmf.getKeyManagers(), null, null);
            // 从上述SSLContext对象中得到SSLSocketFactory对象
            SSLSocketFactory ssf = sslContext.getSocketFactory();
            // 设置ssl证书
            httpsUrlConnection.setSSLSocketFactory(ssf);
        } else {
            // 创建https请求证书
            TrustManager[] tm = { new WxX509TrustManager() };
            // 创建证书上下文对象
            SSLContext sslContext = SSLContext.getInstance("SSL", "SunJSSE");
            // 初始化证书信息
            sslContext.init(null, tm, new SecureRandom());
            // 从上述SSLContext对象中得到SSLSocketFactory对象
            SSLSocketFactory ssf = sslContext.getSocketFactory();
            // 设置ssl证书
            httpsUrlConnection.setSSLSocketFactory(ssf);
        }
        // 设置header信息
        // httpsUrlConnection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
        // 设置User-Agent信息
        httpsUrlConnection.setRequestProperty("User-Agent",
            "Mozilla/5.0 (Windows NT 6.3; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/33.0.1750.146 Safari/537.36");
        // 设置可接受信息
        httpsUrlConnection.setDoOutput(true);
        // 设置可输入信息
        httpsUrlConnection.setDoInput(true);
        // 设置请求方式
        httpsUrlConnection.setRequestMethod(method);
        // 设置连接超时时间
        httpsUrlConnection.setConnectTimeout(CONNECTION_TIMEOUT);
        // 设置请求超时
        httpsUrlConnection.setReadTimeout(READ_TIMEOUT);
        // 设置编码
        httpsUrlConnection.setRequestProperty("Charsert", "UTF-8");
        httpsUrlConnection.setUseCaches(false);
    }

    /**
     * 上传文件
     *
     * @param url 上传地址
     * @param file 上传文件对象
     * @return 服务器上传响应结果
     * @throws Exception 微信操作异常
     */
    public String uploadHttps(String url, File file) throws Exception {
        HttpsURLConnection https = null;
        StringBuffer bufferRes;
        try {
            // 定义数据分隔线
            String BOUNDARY = "----WebKitFormBoundaryiDGnV9zdZA1eM1yL";
            // 创建https请求连接
            https = getHttpsURLConnection(url);
            // 设置header和ssl证书
            setHttpsHeader(https, POST, false, null, null, null);
            // 不缓存
            https.setUseCaches(false);
            // 保持连接
            https.setRequestProperty("connection", "Keep-Alive");
            // 设置文档类型
            https.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + BOUNDARY);

            // 定义输出流
            OutputStream out = null;
            // 定义输入流
            try(DataInputStream dataInputStream = new DataInputStream(new FileInputStream(file));) {
                out = new DataOutputStream(https.getOutputStream());
                byte[] end_data = ("\r\n--" + BOUNDARY + "--\r\n").getBytes();// 定义最后数据分隔线
                StringBuilder sb = new StringBuilder();
                sb.append("--");
                sb.append(BOUNDARY);
                sb.append("\r\n");
                sb.append("Content-Disposition: form-data;name=\"media\";filename=\"").append(file.getName())
                    .append("\"\r\n");
                sb.append("Content-Type:application/octet-stream\r\n\r\n");
                byte[] data = sb.toString().getBytes();
                out.write(data);
                // 读取文件流

                int bytes;
                byte[] bufferOut = new byte[1024];
                while ((bytes = dataInputStream.read(bufferOut)) != -1) {
                    out.write(bufferOut, 0, bytes);
                }
                out.write("\r\n".getBytes()); // 多个文件时，二个文件之间加入这个

                out.write(end_data);
                out.flush();
            } finally {
                if (out != null) {
                    out.close();
                }
            }

            // 定义BufferedReader输入流来读取URL的响应
            InputStream ins = https.getInputStream();
            try(BufferedReader read = new BufferedReader(new InputStreamReader(ins, StandardCharsets.UTF_8))) {
                String valueString;
                bufferRes = new StringBuffer();
                while ((valueString = read.readLine()) != null) {
                    bufferRes.append(valueString);
                }
            } finally {
                if (ins != null) {
                    ins.close();
                }
            }
        } catch (IOException | NoSuchAlgorithmException | KeyManagementException | NoSuchProviderException |
                KeyStoreException | CertificateException | UnrecoverableKeyException ex) {
            throw new Exception(ex.getMessage(), ex);
        } finally {
            if (https != null) {
                // 关闭连接
                https.disconnect();
            }
        }
        return bufferRes.toString();
    }

    public HttpResponse requestHttps(@NonNull String url, String json) throws IOException, GeneralSecurityException {
        HttpResponse response;
        HttpsURLConnection https;
        // 创建https请求连接
        https = getHttpsURLConnection(url);
        // 设置header和ssl证书
        setHttpsHeader(https, POST, false, null, null, null);
        // 保持连接
        https.setRequestProperty("connection", "Keep-Alive");

        https.connect();

        // 获取输出流
        if (StringUtils.isNotEmpty(json)) {
            OutputStream os = https.getOutputStream();
            os.write(json.getBytes("UTF-8"));
            os.close();
        }

        // 获取输入流
        InputStream in = https.getInputStream();

        // 初始化
        response = new HttpResponse();
        // 获取输入流中的Content-Type内容
        response.setContentType(https.getContentType());
        response.setInputStream(new ByteArrayInputStream(IOUtils.toByteArray(in)));
        response.setContentDisposition(https.getHeaderField("Content-disposition"));
        response.setContentLength(https.getHeaderField("Content-Length"));
        response.setHeaders(https.getHeaderFields());
        in.close();
        https.disconnect();
        return response;
    }

    /**
     * 下载附件
     *
     * @param url 附件地址
     * @return 附件对象
     */
    public Attachment downloadHttps(String url) throws IOException, GeneralSecurityException {
        // 定义下载附件对象
        Attachment attachment = null;
        HttpsURLConnection https = null;
        try {
            // 创建https请求连接
            https = getHttpsURLConnection(url);
            // 设置header和ssl证书
            setHttpsHeader(https, POST, false, null, null, null);
            // 保持连接
            https.setRequestProperty("connection", "Keep-Alive");

            https.connect();

            // 获取输入流
            InputStream in = https.getInputStream();

            // 初始化返回附件对象
            attachment = new Attachment();
            // 获取输入流中的Content-Type内容
            String contentType = https.getContentType();
            // 出现错误时，返回错误消息
            if (contentType.contains("text/plain") || contentType.contains("application/json")) {
                // 定义BufferedReader输入流来读取URL的响应
                StringBuilder bufferRes = new StringBuilder();
                try(BufferedReader read = new BufferedReader(new InputStreamReader(in, StandardCharsets.UTF_8))) {
                    String valueString;
                    while ((valueString = read.readLine()) != null) {
                        bufferRes.append(valueString);
                    }
                }
                String textString = bufferRes.toString();

                // 解析错误码
                if (textString.contains("errcode")) {
                    JSONObject result = JSONObject.parseObject(textString);
                    if (result.containsKey("errcode") && result.getIntValue("errcode") != 0) {
                        attachment.setErrCode(result.getInteger("errcode"));
                        attachment.setErrMsg(result.getString("errmsg"));
                        return attachment;
                    }
                }

                if (textString.contains("video_url")) {
                    JSONObject result = JSONObject.parseObject(textString);
                    if (result.containsKey("video_url")) {
                        // 发起get请求获取视频流
                        HttpClient httpClient = new HttpClient();
                        attachment = httpClient.download(result.getString("video_url"));
                    } else {
                        // 未知格式
                        attachment.setErrCode(-1);
                        attachment.setErrMsg(textString);
                    }
                } else {
                    attachment.setErrCode(-1);
                    attachment.setErrMsg(textString);
                }
            } else {
                // 获取数据
                byte[] data = IOUtils.toByteArray(in);

                String ds = https.getHeaderField("Content-disposition");
                // 这里返回的是 attachment; filename=xxxxxxx.mp4
                // 也有可能是 form-data; name="aaa"; filename="xxxxxxxx.mp4"
                String fileName = "";
                if (ds.contains("attachment;")) {
                    fileName = ds.substring(ds.indexOf("filename=") + 10, ds.length());
                } else if (ds.contains("form-data;")) {
                    fileName = ds.substring(ds.indexOf("filename=\"") + 10, ds.length() - 1);
                }
                attachment.setFileName(fileName);
                attachment.setContentLength(https.getHeaderField("Content-Length"));
                attachment.setContentType(https.getHeaderField("Content-Type"));
                attachment.setInputStream(new ByteArrayInputStream(data));//NOSONAR
            }
        } catch (IOException | GeneralSecurityException e) {
            throw e;
        } finally {
            if (https != null) {
                https.disconnect();
            }
        }
        return attachment;
    }
}
