package cn.kinyun.scrm.weixin.sdk.api.shop;

import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.CategoryResp;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.UploadImgResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.*;
import org.springframework.stereotype.Component;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.io.File;
import java.text.MessageFormat;
import java.util.Map;
import java.util.Objects;

/**
 * 微信小商店工具api
 *
 * @author zhangjing
 * @version 1.0
 * @title WxShopUtilAPI
 * @desc 微信小商店工具api
 * @date 2021年12月30日
 * @see <a href="https://developers.weixin.qq.com/miniprogram/dev/platform-capabilities/business-capabilities/ministore/minishopopencomponent/API/register/introduction.html">小商店工具接口</a>
 */
@Slf4j
@Component
public class WxShopUtilAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 上传图片
     */
    @Value("${wx.shop.base.upload}")
    private String uploadPost;

    /**
     * 获取类目详情
     */
    @Value("${wx.shop.base.category}")
    private String categoryPost;

    /**
     * 上传图片
     *
     * @param accessToken 接口调用凭证
     * @param uploadType  上传类型：0-二进制流，1-图片url
     * @param respType    返回数据类型：0-media_id和pay_media_id（此类图片上传大小限制2MB），1-图片临时链接（商品信息相关图片请务必使用此参数得到临时链接）
     * @param imgUrl      上传到OSS后的图片地址, 和file不能同时为空
     * @param file        上传的文件, 和imgUrl不能同时为空
     * @throws WeixinException
     */
    public UploadImgResp uploadImg(@NonNull String accessToken, @NonNull Integer uploadType, @NonNull Integer respType, String imgUrl, File file)
            throws WeixinException {
        log.info("uploadImg with uploadType={},respType={},imgUrl={}", uploadType, respType, imgUrl);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(uploadType >= 0 && uploadType <= 1, "uploadType不能为空,且取值为[0,1]");
        Preconditions.checkArgument(respType >= 0 && respType <= 1, "respType不能为空,且取值为[0,1]");
        Preconditions.checkArgument(ObjectUtils.anyNotNull(imgUrl, file), "imgUrl和file不能同时为空");

        ResponseEntity<UploadImgResp> resp = null;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        if (file != null && StringUtils.isEmpty(imgUrl)) {
            // 构造请求体
            FileSystemResource resource = new FileSystemResource(file);

            MultiValueMap<String, Object> param = new LinkedMultiValueMap<>();
            param.add("media", resource);
            String url = MessageFormat.format(uploadPost, accessToken, uploadType, respType);
            HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<MultiValueMap<String, Object>>(param);
            resp = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                    UploadImgResp.class);
        } else {
            // 构造请求体
            Map<String, Object> params = Maps.newHashMap();
            params.put("img_url", imgUrl);

            HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<Map<String, Object>>(params, headers);

            // 发送请求
            String url = MessageFormat.format(uploadPost, accessToken, uploadType, respType);
            resp = restTemplate.postForEntity(url, httpEntity, UploadImgResp.class);
        }
        UploadImgResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取类目详情
     *
     * @param accessToken 接口调用凭证
     * @param fCatId      父类目ID，可先填0获取根部类目
     * @return
     */
    public CategoryResp getCategory(@NonNull String accessToken, @NonNull Long fCatId) {
        log.info("getCategory with fCatId={}", fCatId);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(fCatId), "fCatId不能为空");

        ResponseEntity<CategoryResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        Map<String, Object> params = Maps.newHashMap();
        params.put("f_cat_id", fCatId);
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<Map<String, Object>>(params, headers);

        // 发送请求
        String url = MessageFormat.format(categoryPost, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, CategoryResp.class);

        CategoryResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
