package cn.kinyun.scrm.weixin.sdk.api.shop;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.shop.req.*;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.OrderDetailResp;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.OrderGetListResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Objects;

/**
 * @author zhangjing
 * @date 2022/1/4 4:31 下午
 */
@Slf4j
@Component
public class WxShopOrderAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 订单接口-获取订单列表 POST
     */
    @Value("${wx.shop.order.get_list}")
    private String shopOrderGetListUrl;

    /**
     * 订单接口-获取订单详情 POST
     */
    @Value("${wx.shop.order.get}")
    private String shopOrderGetUrl;

    /**
     * 订单接口-搜索订单 POST
     */
    @Value("${wx.shop.order.search}")
    private String shopOrderSearchUrl;

    /**
     * 订单接口-修改订单价格 POST
     */
    @Value("${wx.shop.order.change_order_price}")
    private String shopOrderChangeOrderPriceUrl;

    /**
     * 订单接口-修改订单备注 POST
     */
    @Value("${wx.shop.order.change_merchant_notes}")
    private String shopOrderChangeMerchantNotesUrl;

    /**
     * 获取订单列表
     *
     * @param accessToken
     * @param req
     * @return
     */
    public OrderGetListResp orderGetList(@NonNull String accessToken, @NonNull OrderGetListReq req) {
        log.info("orderGetList with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<OrderGetListResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<OrderGetListReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopOrderGetListUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, OrderGetListResp.class);

        OrderGetListResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取订单详情
     *
     * @param accessToken
     * @param req
     * @return
     */
    public OrderDetailResp orderGetDetail(@NonNull String accessToken, @NonNull OrderDetailReq req) {
        log.info("orderGetDetail with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<OrderDetailResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<OrderDetailReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopOrderGetUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, OrderDetailResp.class);

        OrderDetailResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 搜索订单
     *
     * @param accessToken
     * @param req
     * @return
     */
    public OrderGetListResp orderSearch(@NonNull String accessToken, @NonNull OrderSearchReq req) {
        log.info("orderSearch with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<OrderGetListResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<OrderSearchReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopOrderSearchUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, OrderGetListResp.class);

        OrderGetListResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 修改订单价格
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode modOrderPrice(@NonNull String accessToken, @NonNull OrderPriceChangeReq req) {
        log.info("modOrderPrice with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<OrderPriceChangeReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopOrderChangeOrderPriceUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 修改订单备注
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode modOrderNote(@NonNull String accessToken, @NonNull OrderChangeNoteReq req) {
        log.info("modOrderNote with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<OrderChangeNoteReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopOrderChangeMerchantNotesUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
