/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api;

import cn.kinyun.scrm.weixin.sdk.entity.material.resp.UploadImgResp;
import cn.kinyun.scrm.weixin.sdk.entity.media.Attachment;
import cn.kinyun.scrm.weixin.sdk.entity.media.MediaResult;
import cn.kinyun.scrm.weixin.sdk.entity.message.resp.VideoMsg;
import cn.kinyun.scrm.weixin.sdk.enums.WxMediaType;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import cn.kinyun.scrm.weixin.sdk.utils.HttpsClient;
import com.google.common.base.Preconditions;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.io.File;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.text.MessageFormat;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 微信素材接口，用于上传素材
 * 
 * @title WxMediaAPI
 * @desc 微信素材接口
 * @author yanmaoyuan
 * @date 2019年4月28日
 * @version 1.0
 */
@Slf4j
@Component
public class WxMediaAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 上传群发视频素材 POST
     */
    @Value("${wx.media.uploadvideo}")
    private String wxMediaUploadVideo;

    /**
     * 上传图文消息内的图片获取URL POST
     */
    @Value("${wx.media.uploadimg}")
    private String wxMediaUploadImg;

    /**
     * 新增临时素材 POST/FORM
     */
    @Value("${wx.media.upload}")
    private String wxMediaUpload;

    /**
     * 获取临时素材 GET
     */
    @Value("${wx.media.get}")
    private String wxMediaGet;

    /**
     * 高清语音素材获取接口 GET
     */
    @Value("${wx.media.get.jssdk}")
    private String wxMediaGetJssdk;

    /**
     * 上传视频素材
     * 
     * @param accessToken 接口调用凭证
     * @param videoMsg 视频素材
     * @return
     * @throws WeixinException
     */
    public MediaResult uploadVideo(@NonNull String accessToken, @NonNull VideoMsg videoMsg) throws WeixinException {
        log.info("upload video to get mpvideo. videoMsg={}", videoMsg);

        Preconditions.checkArgument(videoMsg.getMediaId() != null, "素材ID为空");
        Preconditions.checkArgument(StringUtils.isNoneBlank(videoMsg.getTitle()), "视频标题为空");
        Preconditions.checkArgument(StringUtils.isNoneBlank(videoMsg.getDescription()), "视频描述为空");

        /**
         * 构造请求头
         */
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        /**
         * 构造请求体
         */
        HttpEntity<VideoMsg> request = new HttpEntity<VideoMsg>(videoMsg, headers);

        /**
         * 发送请求
         */
        String url = MessageFormat.format(wxMediaUploadVideo, accessToken);
        ResponseEntity<MediaResult> response = restTemplate.postForEntity(url, request, MediaResult.class);

        MediaResult result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码

        return result;
    }

    /**
     * 上传图文消息内的图片并返回URL
     *
     * @param accessToken
     * @param file 文件
     * @return
     */
    public UploadImgResp uploadImg(@NonNull String accessToken, @NonNull File file) {

        FileSystemResource resource = new FileSystemResource(file);
        MultiValueMap<String, Object> param = new LinkedMultiValueMap<>();
        param.add("media", resource);
        param.add("filename", file.getName());
        HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<MultiValueMap<String, Object>>(param);
        ResponseEntity<UploadImgResp> resp =
            restTemplate.exchange(wxMediaUploadImg, HttpMethod.POST, httpEntity, UploadImgResp.class, accessToken);

        UploadImgResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 新增临时素材
     *
     * @param accessToken
     * @param type 文件类型
     * @param file 文件
     * @return
     */
    public MediaResult upload(@NonNull String accessToken, @NonNull WxMediaType type, @NonNull File file) {
        FileSystemResource resource = new FileSystemResource(file);
        MultiValueMap<String, Object> param = new LinkedMultiValueMap<>();
        param.add("media", resource);
        param.add("filename", file.getName());
        HttpEntity<MultiValueMap<String, Object>> httpEntity = new HttpEntity<MultiValueMap<String, Object>>(param);
        ResponseEntity<MediaResult> resp = restTemplate.exchange(wxMediaUpload, HttpMethod.POST, httpEntity,
            MediaResult.class, accessToken, type.getValue());
        MediaResult result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取临时素材接口
     *
     * @param accessToken
     * @param mediaId 素材ID
     * @return
     * @throws GeneralSecurityException 
     * @throws IOException 
     */
    public Attachment get(@NonNull String accessToken, @NonNull String mediaId) throws WeixinException, IOException, GeneralSecurityException {
        String url = MessageFormat.format(wxMediaGet, accessToken, mediaId);
        HttpsClient client = new HttpsClient();
        Attachment result = client.downloadHttps(url);
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取高清语音素材接口
     *
     * @param accessToken
     * @param mediaId 素材ID
     * @return
     * @throws GeneralSecurityException 
     * @throws IOException 
     */
    public Attachment getJssdk(@NonNull String accessToken, @NonNull String mediaId) throws WeixinException, IOException, GeneralSecurityException {
        String url = MessageFormat.format(wxMediaGetJssdk, accessToken, mediaId);
        HttpsClient client = new HttpsClient();
        Attachment result = client.downloadHttps(url);
        WeixinException.isSuccess(result);
        return result;
    }
}