package cn.kinyun.scrm.weixin.sdk.api.component;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.component.*;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.collect.Maps;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import javax.servlet.http.HttpServletRequest;
import java.text.MessageFormat;
import java.util.Map;

/**
 * @author : shanyu
 * @version V1.0
 * @Project: weixin-platform
 * @Package com.kuaike.weixin.auth
 * @Description: 第三方平台授权接口
 * @date Date : 2019年04月22日 14:31
 */
@Slf4j
@Component
public class ComponentAuthAPI {

    @Autowired
    private RestTemplate restTemplate;

    @Value("${wx.component.access.token}")
    private String wxComponentAccessToken;

    @Value("${wx.component.preAuthCode}")
    private String wxComponentPreAuthCode;

    @Value("${wx.component.loginpage}")
    private String wxComponentLoginPage;

    @Value("${wx.component.bindcomponent}")
    private String wxComponentBindComponent;

    @Value("${wx.component.apiQueryAuth}")
    private String wxComponentApiQueryAuth;

    @Value("${wx.component.apiGetAuthorizerInfo}")
    private String wxComponentApiGetAuthorizerInfo;

    @Value("${wx.component.apiAuthorizerToken}")
    private String wxComponentApiAuthorizerToken;

    @Value("${wx.component.apiGetAuthorizerOption}")
    private String wxComponentApiGetAuthorizerOption;

    @Value("${wx.component.apiSetAuthorizerOption}")
    private String wxComponentApiSetAuthorizerOption;

    /**
     * 公众号授权之后微信回调返回授权码
     *
     * @param request
     * @return
     */
    public String callBackAuthCode(@NonNull HttpServletRequest request) {
        return request.getParameter("auth_code");
    }

    /**
     * 获取ComponentAccessToken
     *
     * @param appId 第三方平台appid
     * @param appSecret 第三方平台appsecret
     * @param verifyTicket 微信后台推送的ticket
     * @return
     */
    public AccessToken componentAccessToken(@NonNull String appId, @NonNull String appSecret,
                                            @NonNull String verifyTicket) throws WeixinException {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = Maps.newHashMap();
        params.put("component_appid", appId);
        params.put("component_appsecret", appSecret);
        params.put("component_verify_ticket", verifyTicket);
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);
        ResponseEntity<AccessToken> response =
            restTemplate.postForEntity(wxComponentAccessToken, request, AccessToken.class);
        AccessToken result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取预授权码
     *
     * @param accessToken 第三方平台access_token
     * @param componetAppId 第三方平台appid
     * @return
     */
    public PreAuthCode preAuthCode(@NonNull String accessToken, @NonNull String componetAppId) throws WeixinException {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = Maps.newHashMap();
        params.put("component_appid", componetAppId);
        String url = MessageFormat.format(wxComponentPreAuthCode, new Object[] { accessToken });
        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<PreAuthCode> response = restTemplate.postForEntity(url, request, PreAuthCode.class);
        PreAuthCode result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 授权注册页面
     *
     * @param componentAppId 第三方平台appId
     * @param preAuthCode 预授权码
     * @param redirectUri 回调
     * @param authType 要授权的账号类型（1，公众号；2，小程序；3，公众号和小程序）
     * @return
     */
    public String loginPage(@NonNull String componentAppId, @NonNull String preAuthCode, @NonNull String redirectUri,
        int authType) {
        return MessageFormat.format(wxComponentLoginPage, componentAppId, preAuthCode, redirectUri, authType);
    }

    /**
     * 快速授权链接
     *
     * @param appId 第三方appId
     * @param preAuthCode 预授权码
     * @param redirectUri 回调
     * @param authType 要授权的账号类型（1，公众号；2，小程序；3，公众号和小程序）
     * @return
     */
    public String bindcomponent(@NonNull String appId, @NonNull String preAuthCode, @NonNull String redirectUri,
        int authType) {
        return MessageFormat.format(wxComponentBindComponent, appId, preAuthCode, redirectUri, authType);
    }

    /**
     * 根据授权码获取调用凭证和授权信息
     *
     * @param accessToken 第三方平台token
     * @param componetAppId 第三方平台appId
     * @param authorizationCode 公众号授权成功之后通过回调接口获取
     * @return
     */
    public AuthInfoResp apiQueryAuth(@NonNull String accessToken, @NonNull String componetAppId,
                                     @NonNull String authorizationCode) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = Maps.newHashMap();
        params.put("component_appid", componetAppId);
        params.put("authorization_code", authorizationCode);
        String url = MessageFormat.format(wxComponentApiQueryAuth, new Object[] { accessToken });
        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<AuthInfoResp> response = restTemplate.postForEntity(url, request, AuthInfoResp.class);
        log.info("apiQueryAuth result={}",response.getBody());
        AuthInfoResp result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取授权方的帐号基本信息
     *
     * @param accessToken 第三方平台token
     * @param componetAppId 第三方appId
     * @param authorierAppId 授权方appId
     * @return
     */
    public AuthorizerInfoResp apiGetAuthorizerInfo(@NonNull String accessToken, @NonNull String componetAppId,
                                                   @NonNull String authorierAppId) throws WeixinException {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = Maps.newHashMap();
        params.put("component_appid", componetAppId);
        params.put("authorizer_appid", authorierAppId);
        String url = MessageFormat.format(wxComponentApiGetAuthorizerInfo, new Object[] { accessToken });
        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<AuthorizerInfoResp> response =
            restTemplate.postForEntity(url, request, AuthorizerInfoResp.class);
        AuthorizerInfoResp result = response.getBody();
        log.info("apiGetAuthorizerInfo with result={}", result);
        WeixinException.isSuccess(result);
        return result;

    }

    /**
     * 获取（刷新）授权公众号或小程序的接口调用凭据
     *
     * @param accessToken 第三方平台token
     * @param componentAppId 第三方平台appId
     * @param authAppId 授权方appId
     * @param refreshToken 授权方刷新令牌
     * @return
     */
    public AuthorizerToken refreshAuthorizerToken(@NonNull String accessToken, @NonNull String componentAppId,
                                                  @NonNull String authAppId, @NonNull String refreshToken) throws WeixinException {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = Maps.newHashMap();
        params.put("component_appid", componentAppId);
        params.put("authorizer_appid", authAppId);
        params.put("authorizer_refresh_token", refreshToken);
        String url = MessageFormat.format(wxComponentApiAuthorizerToken, new Object[] { accessToken });
        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<AuthorizerToken> response = restTemplate.postForEntity(url, request, AuthorizerToken.class);
        AuthorizerToken result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取授权方的选项设置信息
     *
     * @param accessToken 第三方平台token
     * @param componentId 第三方平台appId
     * @param authorizerAppId 授权公众号或小程序appId
     * @param optionName 选项名称
     * @return
     */
    public AuthorizerOption getAuthorizerOption(@NonNull String accessToken, @NonNull String componentId,
        @NonNull String authorizerAppId, @NonNull String optionName) throws WeixinException {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = Maps.newHashMap();
        params.put("component_appid", componentId);
        params.put("authorizer_appid", authorizerAppId);
        params.put("option_name", optionName);
        String url = MessageFormat.format(wxComponentApiGetAuthorizerOption, new Object[] { accessToken });
        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<AuthorizerOption> response = restTemplate.postForEntity(url, request, AuthorizerOption.class);
        AuthorizerOption result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 设置授权方的选项值
     *
     * @param accessToken
     * @param componentId
     * @param authorizerAppId
     * @param optionName
     * @param optionValue
     * @return
     */
    public void setAuthorizerOption(@NonNull String accessToken, @NonNull String componentId,
        @NonNull String authorizerAppId, @NonNull String optionName, @NonNull String optionValue)
        throws WeixinException {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        Map<String, Object> params = Maps.newHashMap();
        params.put("component_appid", componentId);
        params.put("authorizer_appid", authorizerAppId);
        params.put("option_name", optionName);
        params.put("option_value", optionValue);
        String url = MessageFormat.format(wxComponentApiSetAuthorizerOption, new Object[] { accessToken });
        HttpEntity<Map<String, Object>> request = new HttpEntity<>(params, headers);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);
        ErrorCode result = response.getBody();
        WeixinException.isSuccess(result);
    }
}
