/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api.analysis;

import cn.kinyun.scrm.weixin.sdk.entity.analysis.AnalysisReq;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.user.UserCumulate;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.user.UserSummary;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 微信用户数据分析接口
 * 
 * <p>
 * 在公众号登录授权机制的权限集划分中，用户分析数据接口属于用户管理权限。
 * </p>
 * 
 * @see <a href="https://developers.weixin.qq.com/doc/offiaccount/Analytics/User_Analysis_Data_Interface.html">数据统计-用户分析</a>
 * @title WxUserAnalysisAPI
 * @author yanmaoyuan
 * @date 2019年9月24日
 * @version 1.0
 */
@Slf4j
@Component
public class WxUserAnalysisAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 获取用户增减数据 POST
     */
    @Value("${wx.analysis.getusersummary}")
    private String wxAnalysisGetUserSummary;

    /**
     * 获取累计用户数据 POST
     */
    @Value("${wx.analysis.getusercumulate}")
    private String wxAnalysisGetUserCumulate;

    /**
     * 获取用户增减数据
     * 
     * <p>
     * 最大时间跨度:7天
     * <p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UserSummary getUserSummary(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get user summary with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUserSummary, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UserSummary> response = restTemplate.postForEntity(url, request, UserSummary.class);

        UserSummary result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取累计用户数据
     * 
     * <p>
     * 最大时间跨度:7天
     * <p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UserCumulate getUserCumulate(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get user cumulate with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUserCumulate, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UserCumulate> response = restTemplate.postForEntity(url, request, UserCumulate.class);

        UserCumulate result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}