/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api;

import cn.kinyun.scrm.weixin.sdk.entity.basic.Ticket;
import cn.kinyun.scrm.weixin.sdk.entity.jssdk.UrlSignature;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import cn.kinyun.scrm.weixin.sdk.utils.aes.AesException;
import cn.kinyun.scrm.weixin.sdk.utils.aes.SHA1;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;
import java.util.UUID;

/**
 * 微信JSSDK接口
 * 
 * @title WxJsSdkAPI
 * @desc 微信JSSDK接口
 * @author yanmaoyuan
 * @date 2019年4月29日
 * @version 1.0
 */
@Slf4j
@Component
public class WxJsSdkAPI {

    @Autowired
    private WxBasicAPI wxBasicApi;

    /**
     * 获得jsapi的临时票据
     * 
     * @param accessToken 接口调用凭证
     * @return jsapi临时票据
     * @throws WeixinException
     */
    public Ticket getJsApiTicket(@NonNull String accessToken)
        throws WeixinException {
        return wxBasicApi.getTicket(accessToken, "jsapi");
    }

    /**
     * 使用jsapi_ticket和appid对url进行签名，用于JSSDK认证。
     * 
     * @param url 需要签名的url
     * @param jsApiTicket jsapi临时票据
     * @param appId 微信公众号的appid
     * @return 签名结果
     */
    public UrlSignature signature(@NonNull String url, @NonNull String jsApiTicket, @NonNull String appId) {
        log.info("signature for url={}", url);

        String nonceStr = UUID.randomUUID().toString();
        String timestamp = String.valueOf(System.currentTimeMillis() / 1000);

        // 步骤1. 对所有待签名参数按照字段名的ASCII码从小到大排序（字典序）后，使用URL键值对的格式（即key1=value1&key2=value2…）拼接成字符串

        Map<String, String> paramMap = new TreeMap<String, String>();
        paramMap.put("jsapi_ticket", jsApiTicket);
        paramMap.put("noncestr", nonceStr);
        paramMap.put("timestamp", timestamp);
        paramMap.put("url", url);

        StringBuilder sb = new StringBuilder();
        for (Entry<String, String> entry : paramMap.entrySet()) {
            sb.append('&').append(entry.toString());
        }

        // 步骤2. 对字符串进行sha1签名，得到signature
        String signature = null;
        try {
            signature = SHA1.sha1(sb.toString());
        } catch (AesException e) {
            log.error("签名失败", e);
        }

        // 生成返回结果
        UrlSignature sign = new UrlSignature();
        sign.setAppId(appId);
        sign.setNonceStr(nonceStr);
        sign.setTimestamp(timestamp);
        sign.setSignature(signature);
        sign.setUrl(url);

        return sign;
    }
}