package com.kuaike.scrm.common.perm.service;

import com.kuaike.common.annotation.LoginNeedless;
import com.kuaike.common.annotation.MethodPermission;
import com.kuaike.common.annotation.ModulePremission;
import com.kuaike.scrm.common.perm.dto.PermissionDto;
import com.kuaike.scrm.common.perm.utils.PermUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.condition.PatternsRequestCondition;
import org.springframework.web.servlet.mvc.condition.RequestMethodsRequestCondition;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import java.util.*;

@Slf4j
@Component
public class PermScanner implements ApplicationContextAware {

    // 子系统的权限前缀名
    // 例如：CRM系统 crm、商品订单系统 trade
    // 主系统 scrm 是没有名字的，默认为空字符串
    @Value("${permission.prefix:}")
    private String prefix;

    // 内网服务之间调用扫描结果时，需要传入token来证明身份。
    @Value("${permission.token:}")
    private String token;

    private RequestMappingHandlerMapping handlerMapping;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        handlerMapping = applicationContext.getBean("requestMappingHandlerMapping", RequestMappingHandlerMapping.class);
    }

    /**
     * 扫描当前应用中配置的所有接口权限
     *
     * @return 扫描到的权限结果列表
     */
    public List<PermissionDto> scanAllPerms() {
        return scanAllPerms(token);
    }

    /**
     * 扫描当前应用中配置的所有接口权限
     *
     * @param token 内网鉴权token，用于避免被其他人调用
     * @return 扫描到的权限结果列表
     */
    public List<PermissionDto> scanAllPerms(String token) {
        log.info("scan all permissions");
        if (!Objects.equals(token, this.token)) {
            log.warn("invalid token:{}", token);
            throw new IllegalArgumentException("Invalid token");
        }

        List<PermissionDto> result = new ArrayList<>(512);

        Map<RequestMappingInfo, HandlerMethod> handlerMethods = handlerMapping.getHandlerMethods();

        for (Map.Entry<RequestMappingInfo, HandlerMethod> e : handlerMethods.entrySet()) {//NOSONAR
            RequestMappingInfo info = e.getKey();
            HandlerMethod handlerMethod = e.getValue();

            Class<?> beanType = handlerMethod.getBeanType();

            // 忽略 @LoginNeedless 注解的类
            if (beanType.getAnnotation(LoginNeedless.class) != null) {
                continue;
            }

            // 忽略 @LoginNeedless 注解的方法
            if (handlerMethod.getMethodAnnotation(LoginNeedless.class) != null) {
                continue;
            }

            ModulePremission modulePremission = beanType.getAnnotation(ModulePremission.class);
            if (modulePremission == null) {
                continue;
            }

            MethodPermission methodPermission = handlerMethod.getMethodAnnotation(MethodPermission.class);
            if (methodPermission == null) {
                continue;
            }

            // 构造dto
            PermissionDto perm = new PermissionDto();

            perm.setModuleId(modulePremission.module());
            perm.setModuleName(modulePremission.desc());

            perm.setMethodId(methodPermission.permission());
            perm.setName(methodPermission.desc());

            // 生成权限编码
            String code = PermUtils.getCodeWithPrefix(prefix, perm.getModuleId(), perm.getMethodId());
            perm.setCode(code);

            PatternsRequestCondition condition = info.getPatternsCondition();
            for (String uri : condition.getPatterns()) {
                perm.setPath(PermUtils.getPathWithPrefix(prefix, uri));
            }

            RequestMethodsRequestCondition methodsCondition = info.getMethodsCondition();
            for (RequestMethod method : methodsCondition.getMethods()) {
                perm.setRequestMethod(method.name());
            }

            result.add(perm);
        }

        result.sort(Comparator.comparingLong(it -> {
            // 方法编码应该不会达到一个 int 那么多吧
            return (0xFFFFL & it.getModuleId()) << 32 | it.getMethodId();
        }));

        return result;
    }

    public String getToken() {
        return token;
    }

}
