package com.kuaike.scrm.common.utils;

import com.kuaike.common.annotation.LoginNeedless;
import com.kuaike.scrm.common.dto.CurrentUserInfo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.method.HandlerMethod;
import javax.annotation.PostConstruct;
import javax.servlet.http.HttpSession;
import java.util.Map;
import java.util.HashMap;

/**
 * @author sunwen
 */
@Component
@Slf4j
@SuppressWarnings("pmd:ConfusingTernary")
public class LoginUtils {

    @Autowired
    private RedisTemplate<String, Object> redisTemplate;

    @Value("${spring.redis.key.prefix}")
    private String redisKeyPrefix;

    private static RedisTemplate<String, Object> staticRedisTemplate;

    private static String staticRedisKeyPrefix;

    public static final String CURRENT_USER = "currentUser";
    public static final String HTTP_SESSION = "httpSession";
    public static final String SESSION_ID = "sessionId";
    public static final String USER_ID = "userId";
    public static final String LOGIN_MOBILE = "LOGIN_MOBILE";
    public static final String LOGIN_ACCOUNT_ID = "LOGIN_ACCOUNT_ID";
    /**
     * 调用 "/external" 接口所使用的token会记录在session中，这个值是session中的key。
     */
    public static final String EXTERNAL_TOKEN = "EXTERNAL_TOKEN";

    private static final String LOGIN_KEY = "SCRM_LOGIN_KEY_%s";
    private static final String APP_USER_NAME = "APP_USER_NAME:";
    private static final String APP_REQ_TOKEN = "APP_REQ_TOKEN:";

    private static final ThreadLocal<Map<String, Object>> CONTEXT = new InheritableThreadLocal<>();

    @PostConstruct
    public void init() {
        log.info("Initialize LoginUtils: {}, {}", redisKeyPrefix, redisTemplate);
        staticRedisTemplate = redisTemplate;//NOSONAR
        staticRedisKeyPrefix = redisKeyPrefix;//NOSONAR
    }

    public static Map<String, Object> getContext() {
        Map<String, Object> map = CONTEXT.get();
        if (map == null) {
            map = new HashMap<>();
            CONTEXT.set(map);
        }
        return map;
    }

    public static RedisTemplate<String, Object> getRedisTemplate() {
        return staticRedisTemplate;
    }


    public static void setSession(HttpSession httpSession) {
        getContext().put(HTTP_SESSION, httpSession);
        // 为了兼容老代码，将sessionId放入context中
        getContext().put(SESSION_ID, httpSession.getId());
        getContext().put(LOGIN_ACCOUNT_ID, httpSession.getAttribute(LOGIN_ACCOUNT_ID));
        getContext().put(LOGIN_MOBILE, httpSession.getAttribute(LOGIN_MOBILE));
        getContext().put(EXTERNAL_TOKEN, httpSession.getAttribute(EXTERNAL_TOKEN));
    }

    public static HttpSession getSession() {
        Object val = getContext().get(HTTP_SESSION);
        return val == null ? null : (HttpSession) val;
    }

    public static void setCurrentUser(CurrentUserInfo user) {
        getContext().put(CURRENT_USER, user);
    }

    public static CurrentUserInfo getCurrentUser() {
        Object val = getContext().get(CURRENT_USER);
        return val == null ? null : (CurrentUserInfo) val;
    }

    public static Long getCurrentUserId() {
        CurrentUserInfo loginUser = getCurrentUser();
        if (loginUser != null) {
            return loginUser.getId();
        } else {
            return null;
        }
    }

    public static String getCurrentUserCorpId() {
        CurrentUserInfo loginUser = getCurrentUser();
        if (loginUser != null) {
            return loginUser.getCorpId();
        } else {
            return null;
        }
    }

    public static Long getCurrentUserBizId() {
        CurrentUserInfo loginUser = getCurrentUser();
        if (loginUser != null) {
            return loginUser.getBizId();
        } else {
            return null;
        }
    }

    public static void clear() {
        CONTEXT.remove();
    }

    public static void initCurrentUser(RedisTemplate<String, Object> redisTemplate, String redisKeyPrefix) {
		String key = getUserInfoKey(redisKeyPrefix, getLoginAccountId());
        Object cache = redisTemplate.opsForValue().get(key);
        if (cache != null) {
            if (cache instanceof CurrentUserInfo) {
                setCurrentUser((CurrentUserInfo) cache);
            } else {
                log.error("The user is not front system.");
            }
        }
	}

    public static void initRpcCurrentUser(String sessionId, Long userId) {
		String key = getUserInfoKey(staticRedisKeyPrefix, sessionId, userId);
        Object cache = staticRedisTemplate.opsForValue().get(key);
        if (cache != null) {
            if (cache instanceof CurrentUserInfo) {
                setCurrentUser((CurrentUserInfo) cache);
            } else {
                log.error("The user is not front system.");
            }
        }
	}

    /**
     * 获取员工信息key
     * @param redisKeyPrefix 缓存key的前缀
     * @param userId 用户ID
     * @return 完整的缓存Key
     */
    public static String getUserInfoKey(String redisKeyPrefix, Long userId) {
        return redisKeyPrefix + getSessionId() + String.format(LOGIN_KEY, userId);
    }

    private static String getUserInfoKey(String redisKeyPrefix, String sessionId, Long userId) {
        return redisKeyPrefix + sessionId + String.format(LOGIN_KEY, userId);
    }

    public static void deleteCurrentUser(RedisTemplate<String, Object> redisTemplate, String redisKeyPrefix){
    	String key = getUserInfoKey(redisKeyPrefix, getLoginAccountId());
        redisTemplate.delete(key);
    }

    public static boolean hasLoginNeedless(Object handler) {
        if (!(handler instanceof HandlerMethod)) {
            return false;
        }
        HandlerMethod handlerMethod = (HandlerMethod) handler;
        Class<?> type = handlerMethod.getBeanType();

        // 判断是否需有loginNeedless标记
        LoginNeedless loginNeedless = type.getAnnotation(LoginNeedless.class);
        if (loginNeedless == null) {
            loginNeedless = type.getSuperclass().getAnnotation(LoginNeedless.class);
        }
        if (loginNeedless == null) {
            loginNeedless = handlerMethod.getMethodAnnotation(LoginNeedless.class);
        }
        return loginNeedless != null;
    }

    public static String getSessionId() {
        Object val = getContext().get(SESSION_ID);
        return val == null ? null : (String) val;
    }

    public static void setSessionId(String sessionId) {
        getContext().put(SESSION_ID, sessionId);
    }

    public static Long getLoginAccountId() {
        Object val = getContext().get(LOGIN_ACCOUNT_ID);
        return val == null ? null : (Long) val;
    }

    public static void setLoginAccountId(Long userId) {
        getContext().put(LOGIN_ACCOUNT_ID, userId);

        HttpSession session = getSession();
        if (session != null) {
            session.setAttribute(LOGIN_ACCOUNT_ID, userId);
        }
    }

    public static String getLoginMobile() {
        Object val = getContext().get(LOGIN_MOBILE);
        return val == null ? null : (String) val;
    }

    public static void setLoginMobile(String mobile) {
        getContext().put(LOGIN_MOBILE, mobile);

        HttpSession session = getSession();
        if (session != null) {
            session.setAttribute(LOGIN_MOBILE, mobile);
        }
    }

    public static String getExternalToken() {
        Object val = getContext().get(EXTERNAL_TOKEN);
        return val == null ? null : (String) val;
    }

    public static void setExternalToken(String token) {
        getContext().put(EXTERNAL_TOKEN, token);

        HttpSession session = getSession();
        if (session != null) {
            session.setAttribute(EXTERNAL_TOKEN, token);
        }
    }

    public static void clearExternalToken() {
        getContext().remove(EXTERNAL_TOKEN);

        HttpSession session = getSession();
        if (session != null) {
            session.removeAttribute(EXTERNAL_TOKEN);
        }
    }

    public static void logout() {
        getContext().remove(LOGIN_ACCOUNT_ID);
        getContext().remove(LOGIN_MOBILE);
        getContext().remove(EXTERNAL_TOKEN);

        HttpSession session = getSession();
        if (session != null) {
            session.removeAttribute(LOGIN_ACCOUNT_ID);
            session.removeAttribute(LOGIN_MOBILE);
            session.removeAttribute(EXTERNAL_TOKEN);
        }
    }

    public static String buildAppUserNameKey(String redisKeyPrefix, String username) {
        return redisKeyPrefix + APP_USER_NAME + username;
    }

    public static String buildAppReqTokenKey(String redisKeyPrefix, String reqToken) {
        return redisKeyPrefix + APP_REQ_TOKEN + reqToken;
    }
}
