package com.kuaike.trace.utils;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;

import javax.servlet.http.HttpServletRequest;
import java.util.*;

/**
 * @author louis
 * @date 2022/5/25
 */
@Slf4j
@SuppressWarnings("all")
public final class TraceIdUtils {

    private TraceIdUtils() {
    }
    /**
     * 日志配置文件中有一个traceId，这两者保持一致
     */

    public static final String TRACE_ID = "traceId";

    public static final String B_REQUEST_ID = "B-Request-ID";// 一次请求的唯一ID，用于分布式链路追踪。

    /**
     * 设置traceId
     */
    public static void setTraceId(String traceId) {
        if (StringUtils.isNotBlank(traceId)) {
            //将traceId放到MDC中
            MDC.put(TRACE_ID, traceId);
        }
    }
    /**
     * 获取traceId
     */
    public static String getTraceId() {
        //获取
        return MDC.get(TRACE_ID);

    }
    /**
     * 生成traceId
     */
    public static String generateTraceId() {
        return UUID.randomUUID().toString().replace("-", "");
    }

    public static String genAndSetTraceId() {
        String traceId = generateTraceId();
        setTraceId(traceId);
        return traceId;
    }
    public static void removeTraceId() {
        MDC.remove(TRACE_ID);
    }

    public static Map<String,String> getMDCContactMap() {
       return MDC.getCopyOfContextMap();
    }

    /**
     * 异步情况设置全局traceIdMap信息
     * 如果有异步操作，请配合 {@link TraceIdUtils#removeTraceId()} 使用
     *
     * @param contextMap
     *
     */
    public static void setContextMap(Map<String, String> contextMap) {
        if (MapUtils.isNotEmpty(contextMap)) {
            MDC.setContextMap(contextMap);
        } else {
            MDC.clear();
        }

    }

    public static void generateTraceIdToRequest(HttpServletRequest request) {
        try {
            String traceId = request.getHeader(B_REQUEST_ID);
            if (Objects.nonNull(traceId)) {
                TraceIdUtils.setTraceId(traceId);
            } else {
                traceId = TraceIdUtils.genAndSetTraceId();
                request.setAttribute(B_REQUEST_ID, traceId);
            }
        } catch (Exception e) {
            String requestURI = request.getRequestURI();
            log.error("生成traceId异常:urI:{}", requestURI, e);
        }
    }

}
