package com.kuaike.scrm.dubbo.filter;

import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.Activate;
import com.alibaba.dubbo.rpc.*;
import com.kuaike.scrm.common.utils.LoginUtils;
import com.kuaike.scrm.common.utils.TraceIdUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

/**
 * 在调用dubbo的rpc接口时，通过隐式传参把当前用户的sessionId传到服务端，以达成RPC服务间的session共享。
 *
 * https://blog.csdn.net/syslijian/article/details/106466933
 *
 */
@Slf4j
@Activate(group = {Constants.CONSUMER})
public class ConsumerTraceFilter implements Filter {

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        try {
            RpcInvocation rpcInvocation = (RpcInvocation) invocation;
            // 设置sessionId
            setSessionAttachment(rpcInvocation);
            Result result = invoker.invoke(invocation);
            setTraceToContext();
            return result;
        } catch (RpcException e) {
            log.error("rpc invoke failed", e);
            return null;
        }
    }

    private void setSessionAttachment(RpcInvocation invocation) {
        if (LoginUtils.getCurrentUser() != null) {
            Long userId = LoginUtils.getCurrentUserId();
            String sessionId = LoginUtils.getSessionId();
            log.debug("attach rpc sessionId:{}, userId:{}", sessionId, userId);
            invocation.setAttachment("sessionId", sessionId);
            invocation.setAttachment("userId", userId.toString());
        }
    }

    private void setTraceToContext() {
        String traceId = TraceIdUtils.getTraceId();
        log.info("consumer setTraceId:{}", traceId);

        if (StringUtils.isNotBlank(traceId)) {
            RpcContext.getContext().setAttachment("traceId", traceId);
        }

    }
}
