package com.kuaike.scrm.dubbo.filter;

import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.Activate;
import com.alibaba.dubbo.rpc.*;
import com.kuaike.scrm.common.utils.LoginUtils;
import com.kuaike.scrm.common.utils.TraceIdUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

/**
 * 在调用dubbo的rpc接口时，通过隐式传参把当前用户的sessionId传到服务端，以达成RPC服务间的session共享。
 *
 * https://blog.csdn.net/syslijian/article/details/106466933
 *
 */
@Slf4j
@Activate(group = {Constants.PROVIDER})
public class ProviderTraceFilter implements Filter {

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        try {
            initSession(invocation);

            getTraceFromContext();

            Result result = invoker.invoke(invocation);
            return result;
        } catch (RpcException e) {
            log.error("调用dubbo 接口异常，errorMsg:{}", e.getMessage());
            throw e;
        } finally {
            TraceIdUtils.removeTraceId();
        }
    }

    protected void initSession(Invocation invocation) {
        String sessionId = invocation.getAttachment("sessionId");
        String userIdStr = invocation.getAttachment("userId");
        if (sessionId == null || userIdStr == null) {
            return;
        }

        Long userId = Long.parseLong(userIdStr);
        log.debug("initialize rpc sessionId:{}, userId:{}", sessionId, userId);

        if (LoginUtils.REDIS_TEMPLATE != null) {
            LoginUtils.initRpcCurrentUser(sessionId, userId);
            log.debug("current user bizId:{}", LoginUtils.getCurrentUserBizId());
        } else {
            log.warn("redisTemplate is null");
        }
    }

    protected void getTraceFromContext() {
        String traceId = RpcContext.getContext().getAttachment(TraceIdUtils.TRACE_ID);
        log.debug("provider receive traceId:{}", traceId);
        if (StringUtils.isNotBlank(traceId)) {
            TraceIdUtils.setTraceId(traceId);
        }

    }
}
