package com.kuaike.scrm.common.utils;

import com.kuaike.common.annotation.LoginNeedless;
import com.kuaike.scrm.common.dto.CurrentUserInfo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.method.HandlerMethod;
import javax.annotation.PostConstruct;
import javax.servlet.http.HttpSession;

/**
 * @author sunwen
 */
@Component
@Slf4j
@SuppressWarnings("pmd:ConfusingTernary")
public class LoginUtils {

    @Autowired
    private RedisTemplate<String, Object> redisTemplate;

    @Value("${spring.redis.key.prefix}")
    private String redisKeyPrefix;

    @PostConstruct
    public void init() {
        log.info("Initialize LoginUtils: {}, {}", redisKeyPrefix, redisTemplate);
        staticRedisTemplate = redisTemplate;//NOSONAR
        staticRedisKeyPrefix = redisKeyPrefix;//NOSONAR
    }

    private static RedisTemplate<String, Object> staticRedisTemplate;

    private static String staticRedisKeyPrefix;

    private static final String LOGIN_MOBILE = "LOGIN_MOBILE";

    private static final String LOGIN_ACCOUNT_ID = "LOGIN_ACCOUNT_ID";

    private static final String LOGIN_KEY = "SCRM_LOGIN_KEY_%s";

    private static final String APP_USER_NAME = "APP_USER_NAME:";

    private static final String APP_REQ_TOKEN = "APP_REQ_TOKEN:";

    private static final ThreadLocal<CurrentUserInfo> CURRENT_USER = new InheritableThreadLocal<>();

    private static final ThreadLocal<HttpSession> SESSION = new InheritableThreadLocal<>();

    public static RedisTemplate<String, Object> getRedisTemplate() {
        return staticRedisTemplate;
    }

    public static void setCurrentUser(CurrentUserInfo user) {
        CURRENT_USER.set(user);
    }

    public static CurrentUserInfo getCurrentUser() {
        return CURRENT_USER.get();
    }

    public static Long getCurrentUserId() {
        CurrentUserInfo loginUser = CURRENT_USER.get();
        if (loginUser != null) {
            return loginUser.getId();
        } else {
            return null;
        }
    }

    public static String getCurrentUserCorpId() {
        CurrentUserInfo loginUser = CURRENT_USER.get();
        if (loginUser != null) {
            return loginUser.getCorpId();
        } else {
            return null;
        }
    }

    public static Long getCurrentUserBizId() {
        CurrentUserInfo loginUser = CURRENT_USER.get();
        if (loginUser != null) {
            return loginUser.getBizId();
        } else {
            return null;
        }
    }

    public static void setSession(HttpSession httpSession) {
        SESSION.set(httpSession);
    }

    public static HttpSession getSession() {
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        if (attributes == null) {
            return SESSION.get();
        }
        return attributes.getRequest().getSession();
    }

    public static void clear() {
        CURRENT_USER.remove();
        SESSION.remove();
    }

    public static void initCurrentUser(RedisTemplate<String, Object> redisTemplate, String redisKeyPrefix) {
		String key = getUserInfoKey(redisKeyPrefix, getLoginAccountId());
        Object cache = redisTemplate.opsForValue().get(key);
        if (cache != null) {
            if (cache instanceof CurrentUserInfo) {
                setCurrentUser((CurrentUserInfo) cache);
            } else {
                log.error("The user is not front system.");
            }
        }
        redisTemplate.opsForValue().set(getSessionId(), getLoginAccountId());
	}

    public static void initRpcCurrentUser(String sessionId, Long userId) {
		String key = getUserInfoKey(staticRedisKeyPrefix, sessionId, userId);
        Object cache = staticRedisTemplate.opsForValue().get(key);
        if (cache != null) {
            if (cache instanceof CurrentUserInfo) {
                setCurrentUser((CurrentUserInfo) cache);
            } else {
                log.error("The user is not front system.");
            }
        }
	}

    /**
     * 获取员工信息key
     * @param redisKeyPrefix 缓存key的前缀
     * @param userId 用户ID
     * @return 完整的缓存Key
     */
    public static String getUserInfoKey(String redisKeyPrefix, Long userId) {
        return redisKeyPrefix + getSessionId() + String.format(LOGIN_KEY, userId);
    }

    private static String getUserInfoKey(String redisKeyPrefix, String sessionId, Long userId) {
        return redisKeyPrefix + sessionId + String.format(LOGIN_KEY, userId);
    }

    public static void deleteCurrentUser(RedisTemplate<String, Object> redisTemplate, String redisKeyPrefix){
    	String key = getUserInfoKey(redisKeyPrefix, getLoginAccountId());
        redisTemplate.delete(key);
    }

    public static boolean hasLoginNeedless(Object handler) {
        if (!(handler instanceof HandlerMethod)) {
            return false;
        }
        HandlerMethod handlerMethod = (HandlerMethod) handler;
        Class<?> type = handlerMethod.getBeanType();

        // 判断是否需有loginNeedless标记
        LoginNeedless loginNeedless = type.getAnnotation(LoginNeedless.class);
        if (loginNeedless == null) {
            loginNeedless = type.getSuperclass().getAnnotation(LoginNeedless.class);
        }
        if (loginNeedless == null) {
            loginNeedless = handlerMethod.getMethodAnnotation(LoginNeedless.class);
        }
        return loginNeedless != null;
    }

    public static String getSessionId() {
        HttpSession session = getSession();
        if (session != null) {
            return session.getId();
        }
        return null;
    }

    public static Long getLoginAccountId() {
        Long userId = null;
        HttpSession session = getSession();
        if (session != null) {
            userId = (Long) session.getAttribute(LOGIN_ACCOUNT_ID);
        }
        return userId;
    }

    public static void setLoginAccountId(Long userId) {
        HttpSession session = getSession();
        if (session != null) {
            session.setAttribute(LOGIN_ACCOUNT_ID, userId);
        }
    }

    public static void removeLoginAccountId() {
        HttpSession session = getSession();
        if (session != null) {
            session.removeAttribute(LOGIN_ACCOUNT_ID);
            session.removeAttribute(LOGIN_MOBILE);
        }
    }

    public static void setLoginMobile(String mobile) {
        HttpSession session = getSession();
        if (session != null) {
            session.setAttribute(LOGIN_MOBILE, mobile);
        }
    }

    public static String getLoginMobile() {
        String mobile = null;
        HttpSession session = getSession();
        if (session != null) {
            mobile = (String) session.getAttribute(LOGIN_MOBILE);
        }
        return mobile;
    }

    public static String buildAppUserNameKey(String redisKeyPrefix, String username) {
        return redisKeyPrefix + APP_USER_NAME + username;
    }

    public static String buildAppReqTokenKey(String redisKeyPrefix, String reqToken) {
        return redisKeyPrefix + APP_REQ_TOKEN + reqToken;
    }
}
