package com.kuaike.trace.utils;

import org.apache.commons.collections4.MapUtils;
import org.slf4j.MDC;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;
import java.util.stream.Collectors;

/**
 * @author louis
 * @date 2022/8/19
 *
 * 线程池 traceId 包装utils
 */
public final class TraceExecutorWrapUtil {

    private TraceExecutorWrapUtil() {}

    public static <T> Callable<T> wrap(Callable<T> callable) {
        Map<String, String> context = MDC.getCopyOfContextMap();
        return () -> {
            if (MapUtils.isEmpty(context)) {
                MDC.clear();
            } else {
                MDC.setContextMap(context);
            }
            try {
                return callable.call();
            } finally {
                MDC.clear();
            }
        };
    }
    public static Runnable wrap(Runnable runnable) {
        Map<String, String> context = MDC.getCopyOfContextMap();
        return () -> {
            if (MapUtils.isEmpty(context)) {
                MDC.clear();
            } else {
                MDC.setContextMap(context);
            }
            try {
                runnable.run();
            } finally {
                MDC.clear();
            }
        };
    }

    public static void wrapExecute(Executor executor, Runnable runnable) {
        executor.execute(wrap(runnable));
    }

    public static void wrapSubmit(ExecutorService executorService, Runnable runnable) {
        executorService.submit(wrap(runnable));
    }

    public static <T> Future<T> wrapSubmit(ExecutorService executorService, Callable<T> callable) {
        return executorService.submit(wrap(callable));
    }

    public static <T> List<Future<T>> invokeAll(ExecutorService executorService, Collection<? extends Callable<T>> tasks)throws InterruptedException {
        List<Callable<T>> wrapTasks = tasks.stream().map(TraceExecutorWrapUtil::wrap).collect(Collectors.toList());
        return executorService.invokeAll(wrapTasks);
    }

    public static void wrapCompletableFutureRunAsync(Runnable runnable) {
        CompletableFuture.runAsync(wrap(runnable));
    }

    public static void wrapCompletableFutureRunAsync(Runnable runnable, Executor executor) {
        CompletableFuture.runAsync(wrap(runnable), executor);
    }

}