package com.kuaike.scrm.common.utils;

import com.kuaike.common.annotation.LoginNeedless;
import com.kuaike.scrm.common.dto.CurrentUserInfo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.method.HandlerMethod;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpSession;

/**
 * @author sunwen
 */
@Component
@Slf4j
public class LoginUtils {

    @Autowired
    private RedisTemplate<String, Object> redisTemplate;

    @Value("${spring.redis.key.prefix}")
    private String redisKeyPrefix;

    @PostConstruct
    public void init() {
        log.info("Initialize LoginUtils: {}, {}", redisKeyPrefix, redisTemplate);
        REDIS_TEMPLATE = redisTemplate;//NOSONAR
        REDIS_KEY_PREFIX = redisKeyPrefix;//NOSONAR
    }

    public static RedisTemplate<String, Object> REDIS_TEMPLATE;

    public static String REDIS_KEY_PREFIX;

    private final static String LOGIN_MOBILE = "LOGIN_MOBILE";

    private final static String LOGIN_ACCOUNT_ID = "LOGIN_ACCOUNT_ID";

    private final static String LOGIN_KEY = "SCRM_LOGIN_KEY_%s";

    private static final ThreadLocal<CurrentUserInfo> currentUser = new InheritableThreadLocal<>();
    private static final ThreadLocal<HttpSession> SESSION = new InheritableThreadLocal<>();

    public static void setCurrentUser(CurrentUserInfo user) {
        currentUser.set(user);
    }

    public static CurrentUserInfo getCurrentUser() {
        return currentUser.get();
    }

    public static Long getCurrentUserId() {
        CurrentUserInfo loginUser = currentUser.get();
        if (loginUser != null) {
            return loginUser.getId();
        } else {
            return null;
        }
    }

    public static String getCurrentUserCorpId() {
        CurrentUserInfo loginUser = currentUser.get();
        if (loginUser != null) {
            return loginUser.getCorpId();
        } else {
            return null;
        }
    }

    public static Long getCurrentUserBizId() {
        CurrentUserInfo loginUser = currentUser.get();
        if (loginUser != null) {
            return loginUser.getBizId();
        } else {
            return null;
        }
    }

    public static void setSession(HttpSession httpSession) {
        SESSION.set(httpSession);
    }

    public static HttpSession getSession() {
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        if (attributes == null) {
            return SESSION.get();
        }
        return attributes.getRequest().getSession();
    }

    public static void clear() {
        currentUser.remove();
        SESSION.remove();
    }

    /**
     * 获取员工信息key
     * @param redisKeyPrefix 缓存key的前缀
     * @param userId 用户ID
     * @return 完整的缓存Key
     */
    public static String getUserInfoKey(String redisKeyPrefix, Long userId) {
        return redisKeyPrefix + getSessionId() + String.format(LOGIN_KEY, userId);
    }

    public static void initCurrentUser(RedisTemplate<String, Object> redisTemplate, String redisKeyPrefix) {
		String key = getUserInfoKey(redisKeyPrefix, getLoginAccountId());
        Object cache = redisTemplate.opsForValue().get(key);
        if (cache != null) {
            if (cache instanceof CurrentUserInfo) {
                setCurrentUser((CurrentUserInfo) cache);
            } else {
                log.error("The user is not front system.");
            }
        }
	}

    public static void initRpcCurrentUser(String sessionId, Long userId) {
		String key = getUserInfoKey(REDIS_KEY_PREFIX, sessionId, userId);
        Object cache = REDIS_TEMPLATE.opsForValue().get(key);
        if (cache != null) {
            if (cache instanceof CurrentUserInfo) {
                setCurrentUser((CurrentUserInfo) cache);
            } else {
                log.error("The user is not front system.");
            }
        }
	}

    private static String getUserInfoKey(String redisKeyPrefix, String sessionId, Long userId) {
        return redisKeyPrefix + sessionId + String.format(LOGIN_KEY, userId);
    }

    public static void deleteCurrentUser(RedisTemplate<String, Object> redisTemplate, String redisKeyPrefix){
    	String key = getUserInfoKey(redisKeyPrefix, getLoginAccountId());
        redisTemplate.delete(key);
    }

    public static boolean hasLoginNeedless(Object handler) {
        if (!(handler instanceof HandlerMethod)) {
            return false;
        }
        HandlerMethod handlerMethod = (HandlerMethod) handler;
        Class<?> type = handlerMethod.getBeanType();

        // 判断是否需有loginNeedless标记
        LoginNeedless loginNeedless = type.getAnnotation(LoginNeedless.class);
        if (loginNeedless == null) {
            loginNeedless = type.getSuperclass().getAnnotation(LoginNeedless.class);
        }
        if (loginNeedless == null) {
            loginNeedless = handlerMethod.getMethodAnnotation(LoginNeedless.class);
        }
        return loginNeedless != null;
    }

    public static String getSessionId() {
        HttpSession session = getSession();
        if (session != null) {
            return session.getId();
        }
        return null;
    }

    public static Long getLoginAccountId() {
        Long userId = null;
        HttpSession session = getSession();
        if (session != null) {
            userId = (Long) session.getAttribute(LOGIN_ACCOUNT_ID);
        }
        return userId;
    }

    public static void setLoginAccountId(Long userId) {
        HttpSession session = getSession();
        if (session != null) {
            session.setAttribute(LOGIN_ACCOUNT_ID, userId);
        }
    }

    public static void removeLoginAccountId() {
        HttpSession session = getSession();
        if (session != null) {
            session.removeAttribute(LOGIN_ACCOUNT_ID);
            session.removeAttribute(LOGIN_MOBILE);
        }
    }

    public static void setLoginMobile(String mobile) {
        HttpSession session = getSession();
        if (session != null) {
            session.setAttribute(LOGIN_MOBILE, mobile);
        }
    }

    public static String getLoginMobile() {
        String mobile = null;
        HttpSession session = getSession();
        if (session != null) {
            mobile = (String) session.getAttribute(LOGIN_MOBILE);
        }
        return mobile;
    }
}
