package com.baijiayun.duanxunbao.sdk.service.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baijiayun.duanxunbao.sdk.dto.req.*;
import com.baijiayun.duanxunbao.sdk.dto.resp.*;
import com.baijiayun.duanxunbao.sdk.service.BjyPartnerService;
import com.baijiayun.duanxunbao.sdk.service.ScrmOpenUrl;
import com.baijiayun.duanxunbao.sdk.service.SmsService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.TreeMap;

@Slf4j
@Service
public class SmsServiceImpl implements SmsService {

    private static final int SEND_DETAIL_LIMIT = 100;
    private static final String SEND_ID = "send_id";
    private static final String CONTENT = "content";
    private static final String CORP_ID = "corp_id";
    private static final String SMS_SIGN = "sms_sign";

    @Autowired
    private BjyPartnerService bjyPartnerService;


    @Override
    public SmsCheckUrlResp checkUrl(String partnerId, String token, String content) {
        log.info("checkUrl partnerId: {}, token: {}, content:{}", partnerId, token, content);
        Map<String, String> map = getCheckParams(content);
        SmsCheckUrlResp resp = null;
        try {
            JSONObject jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.SMS_CHECK_URL, map);
            JSONObject data = jsonObject.getJSONObject("data");
            if (MapUtils.isEmpty(data)) {
                log.info("get data is empty");
                return null;
            }
            resp = JSONObject.parseObject(data.toJSONString(), SmsCheckUrlResp.class);
        } catch (Exception e) {
            log.error("checkUrl error", e);
        }
        return resp;
    }

    @Override
    public SmsSendDetail noticeSendInfo(String partnerId, String token, SmsNoticeSendInfoReq req) {
        log.info("noticeSendInfo partnerId: {}, token: {}, req:{}", partnerId, token, req);
        Map<String, String> params = new TreeMap<>();
        params.put("msg_id", req.getMsgId());
        params.put("mobile", req.getMobile());
        SmsSendDetail sendDetail = null;
        try {
            JSONObject jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.SMS_NOTICE_SEND_INFO, params);
            JSONObject data = jsonObject.getJSONObject("data");
            if (MapUtils.isEmpty(data)) {
                log.info("get data is empty");
                return null;
            }
            sendDetail = JSONObject.parseObject(data.toJSONString(), SmsSendDetail.class);
        } catch (Exception e) {
            log.error("get notice sms detail error", e);
        }
        return sendDetail;
    }

    @Override
    public JSONObject noticeSend(String partnerId, String token, SmsNoticeSendReq req) {
        log.info("noticeSend partnerId: {}, token: {}, req:{}", partnerId, token, req);
        Map<String, String> params = new TreeMap<>();
        params.put("sms_sign", req.getSmsSign());
        params.put("content", req.getContent());
        params.put("to", req.getTo());
        return bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.SMS_NOTICE_SEND, params);
    }

    @Override
    public JSONObject open(String partnerId, String token, SmsOpenReq req) {
        log.info("open partnerId: {}, token: {}, req:{}", partnerId, token, req);
        Map<String, String> params = getOpenParams(req.getUnitPrice(), req.getPartnerId());
        return bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.SMS_OPEN_URL, params);
    }

    @Override
    public JSONObject sendDetail(String partnerId, String token, SmsSendDetailReq req) {
        log.info("sendDetail partnerId: {}, token: {}, req:{}", partnerId, token, req);
        Map<String, String> params = new TreeMap<>();
        params.put("send_id", req.getSendId());
        params.put("page", String.valueOf(req.getPage()));
        params.put("page_size", String.valueOf(req.getPageSize()));
        return bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.SMS_SEND_DETAIL, params);
    }

    @Override
    public SmsSendResp send(String partnerId, String token, SmsSendReq req) {
        log.info("send partnerId: {}, token: {}, req:{}", partnerId, token, req);
        Map<String, String> params = new TreeMap<>();
        params.put("sms_sign", req.getSmsSign());
        params.put("content", req.getContent());
        params.put("type", req.getType());
        params.put("to", req.getTo());
        params.put("custom_params", req.getCustomParams());

        SmsSendResp sendResp = new SmsSendResp();
        try {
            JSONObject jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.SMS_SEND_URL, params);
            JSONObject data = jsonObject.getJSONObject("data");
            int code = jsonObject.getIntValue("code");
            String msg = jsonObject.getString("msg");
            if (code != 0) {
                sendResp.setMsg(msg);
                return sendResp;
            }
            if (MapUtils.isEmpty(data)) {
                log.info("get data is empty");
                return sendResp;
            }
            sendResp = JSONObject.parseObject(data.toJSONString(), SmsSendResp.class);
        } catch (Exception e) {
            log.error("send error", e);
            sendResp.setMsg(e.getMessage());
        }
        return sendResp;
    }

    @Override
    public void templateDelete(String partnerId, String token, List<String> templateIds) {
        log.info("send partnerId: {}, token: {}, templateId:{}", partnerId, token, templateIds);
        Map<String, String> deleteParams = getTemplateDeleteParams(templateIds);
        bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.SMS_TEMPLATE_DEL, deleteParams);
    }

    @Override
    public SmsTemplateUploadResp templateUpload(String partnerId, String token, String content) {
        log.info("templateUpload partnerId: {}, token: {}, content:{}", partnerId, token, content);
        Map<String, String> uploadParams = getTemplateUploadParams(content);
        SmsTemplateUploadResp resp = null;
        try {
            JSONObject jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.SMS_TEMPLATE_UPLOAD, uploadParams);
            JSONObject data = jsonObject.getJSONObject("data");
            if (MapUtils.isEmpty(data)) {
                log.info("get data is empty");
                return null;
            }
            resp = JSONObject.parseObject(data.toJSONString(), SmsTemplateUploadResp.class);
        } catch (Exception e) {
            log.error("templateUpload error", e);
        }
        return resp;
    }

    @Override
    public List<SmsTemplateInfoResp> templateInfo(String partnerId, String token, List<String> templateIds) {
        log.info("templateInfo partnerId: {}, token: {}, templateIds:{}", partnerId, token, templateIds);
        Map<String, String> params = getTemplateInfoParams(templateIds);
        List<SmsTemplateInfoResp> templateDtos = null;
        try {
            JSONObject jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.SMS_TEMPLATE_INFO, params);
            JSONArray data = jsonObject.getJSONArray("data");
            if (data == null || data.isEmpty()) {
                log.info("get data is empty, templateIds:{}", templateIds);
                return null;
            }
            templateDtos = data.toJavaList(SmsTemplateInfoResp.class);
        } catch (Exception e) {
            log.error("templateInfo error", e);
        }
        return templateDtos;
    }


    private Map<String, String> getOpenParams(int unitPrice, String partnerId) {
        Map<String, String> result = new TreeMap<>();
        result.put("unit_price", String.valueOf(unitPrice));
        result.put("partner_id", partnerId);
        result.put("timestamp", String.valueOf(System.currentTimeMillis()));
        return result;
    }


    private Map<String, String> getCheckParams(String content) {
        Map<String, String> map = new TreeMap<>();
        map.put("content", content);
        return map;
    }

    private Map<String, String> getNoticeSendParams(List<String> phones, String content, String signature) {
        Map<String, String> params = new TreeMap<>();
        params.put(SMS_SIGN, signature);
        params.put(CONTENT, content);
        params.put("to", String.join(",", phones));
        return params;
    }

    private Map<String, String> getSendParams(List<String> phones, String content, String bizNum, String signature) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put(CORP_ID, bizNum);
        Map<String, String> map = new TreeMap<>();
        map.put(SMS_SIGN, signature);
        map.put(CONTENT, content);
        map.put("type", "100003");
        map.put("to", String.join(",", phones));
        map.put("custom_params", jsonObject.toJSONString());
        return map;
    }

    private Map<String, String> getSendOneParams(String phone, String content, String bizNum, String signature) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put(CORP_ID, bizNum);
        Map<String, String> map = new TreeMap<>();
        map.put(SMS_SIGN, signature);
        map.put("type", "100003");
        map.put(CONTENT, content);
        map.put("to", phone);
        map.put("custom_params", jsonObject.toJSONString());
        return map;
    }

    private Map<String, String> getNoticeSendInfoParams(String bjyTaskId, String phone) {
        Map<String, String> params = new TreeMap<>();
        params.put("msg_id", bjyTaskId);
        params.put("mobile", phone);
        return params;
    }

    private Map<String, String> getSendDetailParams(String bjyTaskId, int page, String partnerId) {
        Map<String, String> map = new TreeMap<>();
        map.put("partner_id", partnerId);
        map.put(SEND_ID, bjyTaskId);
        map.put("page", String.valueOf(page));
        map.put("page_size", String.valueOf(SEND_DETAIL_LIMIT));
        map.put("timestamp", String.valueOf(System.currentTimeMillis()));
        return map;
    }


    private Map<String, String> getTemplateUploadParams(String content) {
        Map<String, String> map = new TreeMap<>();
        map.put("content", content);
        return map;
    }

    private Map<String, String> getTemplateDeleteParams(List<String> templateIds) {
        Map<String, String> map = new TreeMap<>();
        map.put("template_ids", StringUtils.join(templateIds, ","));
        return map;
    }

    private Map<String, String> getTemplateInfoParams(List<String> templateIds) {
        Map<String, String> map = new TreeMap<>();
        map.put("template_ids", StringUtils.join(templateIds, ","));
        return map;
    }
}
