package com.baijiayun.duanxunbao.sdk.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.baijiayun.duanxunbao.common.enums.ResultCode;
import com.baijiayun.duanxunbao.common.exception.BusinessException;
import com.baijiayun.duanxunbao.common.utils.JacksonUtil;
import com.baijiayun.duanxunbao.common.utils.MobileUtils;
import com.baijiayun.duanxunbao.sdk.dto.req.*;
import com.baijiayun.duanxunbao.sdk.service.BjyPartnerService;
import com.baijiayun.duanxunbao.sdk.service.CallService;
import com.baijiayun.duanxunbao.sdk.service.ScrmOpenUrl;
import com.baijiayun.duanxunbao.sdk.utils.ApiTypeEnum;
import com.baijiayun.duanxunbao.sdk.utils.VoiceMaterialTypeEnum;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

@Slf4j
@Service
public class CallServiceImpl implements CallService {

    @Autowired
    private BjyPartnerService bjyPartnerService;

    @Override
    public Object callAnswerRegister(String partnerId, String token,  CallRegisterReq req) {
        log.info("callAnswerRegister partnerId: {}, token: {},  reqDto:{}", partnerId, token, req);
        Map<String, String> params = getInRegisterParams(req.getCallType(), req.getCallSheetId());
        Object resp = null;
        try {
            JSONObject jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_ANSWER_REGISTER, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            resp = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return resp;
    }

    @Override
    public Object applyExten(String partnerId, String token,  CallApplyExtenReq req) {
        log.info("applyExten partnerId: {}, token: {},  reqDto:{}", partnerId, token, req);
        Map<String, String> params = getApplyExtenParams(req.getPhone(), req.getUserId(), req.getName());
        Object resp = null;
        try {
            JSONObject jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_APPLY_EXTEN, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            resp = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return resp;
    }

    @Override
    public Object callRegister(String partnerId, String token, CallOutDto dto) {
        log.info("callRegister partnerId: {}, token: {},  dto:{}", partnerId, token, dto);
        Map<String, String> params = getCallParams(dto);
        JSONObject jsonObject = null;
        if (dto.getCallType().equals(ApiTypeEnum.LOCAL.getName())) {
            Preconditions.checkArgument(MobileUtils.isPhone(dto.getPhone()), "请输入11位手机号");
            try {
                jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_CLICK_2_CALL, params);
                int code = jsonObject.getIntValue("code");
                if (code != 0) {
                    throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
                }
                jsonObject = jsonObject.getJSONObject("data");
            } catch (Exception e) {
                log.error("sign error", e);
            }
        } else {
            Preconditions.checkArgument(StringUtils.isNotBlank(dto.getSessionId()), "sessionId不能为空");
            try {
                jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_REGISTER, params);
                int code = jsonObject.getIntValue("code");
                if (code != 0) {
                    throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
                }
                jsonObject = jsonObject.getJSONObject("data");
            } catch (Exception e) {
                log.error("sign error", e);
            }
        }
        return jsonObject;
    }

    @Override
    public Object callRelease(String partnerId, String token,  CallReleaseReq req) {
        log.info("callRelease partnerId: {}, token: {},  req:{}", partnerId, token, req);
        Map<String, String> params = getHangupParams(req.getSessionId());
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_RELEASE, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callOpen(String partnerId, String token,  CallConfig config) {
        log.info("callOpen partnerId: {}, token: {},  config:{}", partnerId, token, config);
        Map<String, String> params = getConfigParams(config);
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_OPEN, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callClose(String partnerId, String token) {
        log.info("callClose partnerId: {}, token: {}", partnerId, token);
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_CLOSE, new HashMap<>());
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callDeleteMember(String partnerId, String token, CallMemberDto dto) {
        log.info("callDeleteMember partnerId: {}, token: {},  dto:{}", partnerId, token, dto);
        Map<String, String> params = getDeleteMemberParams(dto.getUserId());
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_MEMBER_DEL, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callForbiddenMember(String partnerId, String token,  CallForbiddenMemberDto dto) {
        log.info("callForbiddenMember partnerId: {}, token: {},  dto:{}", partnerId, token, dto);
        Map<String, String> params = getForbiddenMemberParams(dto.getStatus(), dto.getUserId());
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_MEMBER_FORBIDDEN, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callAddMember(String partnerId, String token, CallAddMemberDto dto) {
        log.info("callAddMember partnerId: {}, token: {},  dto:{}", partnerId, token, dto);
        Map<String, String> params = getAddMemberParams(dto.getPhone(), dto.getName(), dto.getUserId());
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_MEMBER_ADD, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callEditMember(String partnerId, String token, CallEditMemberDto dto) {
        log.info("callEditMember partnerId: {}, token: {},  dto:{}", partnerId, token, dto);
        Map<String, String> params = getEditMemberParams(dto.getNewPhone(), dto.getUserId());
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_MEMBER_EDIT, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callMemberInfo(String partnerId, String token, CallMemberDto dto) {
        log.info("callMemberInfo partnerId: {}, token: {},  dto:{}", partnerId, token, dto);
        Map<String, String> params = getExtenInfoParams(dto.getUserId());
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_MEMBER_INFO, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callSignInOut(String partnerId, String token, CallSigninOutDto dto) {
        log.info("callSignInOut partnerId: {}, token: {},  dto:{}", partnerId, token, dto);
        Map<String, String> params = getSignInOrOutParams(dto.getExten(), dto.getSignInType());
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_MEMBER_SIGN_IN_OOUT, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callSessionId(String partnerId, String token) {
        log.info("callSessionId partnerId: {}, token: {}", partnerId, token);
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_SESSION_ID, Maps.newHashMap());
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callLinesUrl(String partnerId, String token) {
        log.info("callLinesUrl partnerId: {}, token: {}", partnerId, token);
        Map<String, String> params = Maps.newHashMap();
        // 可不传，约定传0
        params.put("partner_id", "0");
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_LINES, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callVoiceUrl(String partnerId, String token, CallVoiceUrlDto dto) {
        log.info("callVoiceUrl partnerId: {}, token: {}, dto: {}", partnerId, token, dto);
        Map<String, String> params = getVoiceParams(dto.getUrl());
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_VOICE, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callRlExtenList(String partnerId, String token) {
        log.info("callRlExtenList partnerId: {}, token: {}", partnerId, token);
        Map<String, String> params = getRlExtenListParams();
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_RL_EXTEN_LIST, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }

    @Override
    public Object callVoice(String partnerId, String token, CallVoiceParamDto dto) {
        log.info("callVoice partnerId: {}, token: {}, dto: {}", partnerId, token, dto);
        Map<String, String> params = handleVoiceParams(dto.getMobile(), dto.getContent(), dto.getType(), dto.getMaterialParams());
        JSONObject jsonObject = null;
        try {
            jsonObject = bjyPartnerService.bjyPost(partnerId, token, ScrmOpenUrl.CALL_VOICE_CALL, params);
            int code = jsonObject.getIntValue("code");
            if (code != 0) {
                throw new BusinessException(ResultCode.BUSINESS_ERROR, "bjyService 接口调用异常");
            }
            jsonObject = jsonObject.getJSONObject("data");
        } catch (Exception e) {
            log.error("sign error", e);
        }
        return jsonObject;
    }


    private Map<String, String> getInRegisterParams(String callType, String callSheetId) {
        Map<String, String> result = new TreeMap<>();
        result.put("call_type", callType);
        result.put("call_sheet_id", callSheetId);
        return result;
    }


    private Map<String, String> getApplyExtenParams(String phone, Long userId, String name) {
        Map<String, String> result = new TreeMap<>();
        result.put("phone", phone);
        result.put("name", name);
        result.put("user_number", String.valueOf(userId));
        return result;
    }

    private Map<String, String> getHangupParams(String sessionId) {
        Map<String, String> result = new TreeMap<>();
        result.put("session_id", sessionId);
        return result;
    }

    private Map<String, String> getCallParams(CallOutDto callOutDto) {
        Map<String, String> result = new TreeMap<>();
        result.put("tel_a", callOutDto.getUserPhone());
        result.put("tel_b", callOutDto.getPhone());
        result.put("user_number", String.valueOf(callOutDto.getUserId()));
        result.put("call_type", callOutDto.getCallType());
        if (StringUtils.isNotBlank(callOutDto.getSessionId())) {
            result.put("session_id", callOutDto.getSessionId());
        }
        return result;
    }

    private Map<String, String> getConfigParams(CallConfig callConfig) {
        Map<String, String> result = new TreeMap<>();
        result.put("supplier_type", String.valueOf(callConfig.getSupplierType()));
        result.put("callback_host", callConfig.getCallbackHost());
        result.put("call_line", callConfig.getCallLine());
        if (!ObjectUtils.isEmpty(callConfig.getAccountInfo())) {
            result.put("account_info", JacksonUtil.obj2Str(callConfig.getAccountInfo()));
        }
        if (!ObjectUtils.isEmpty(callConfig.getWebCallAccountInfo())) {
            result.put("account_webcall_info", JacksonUtil.obj2Str(callConfig.getWebCallAccountInfo()));
        }
        if (!ObjectUtils.isEmpty(callConfig.getDoubleCallAccountInfo())) {
            result.put("account_double_call_info", JacksonUtil.obj2Str(callConfig.getDoubleCallAccountInfo()));
        }
        if (!ObjectUtils.isEmpty(callConfig.getWebCallFeeInfo())) {
            result.put("webcall_fee_info", JacksonUtil.obj2Str(callConfig.getWebCallFeeInfo()));
        }
        if (!ObjectUtils.isEmpty(callConfig.getCallFeeInfo())) {
            result.put("call_fee_info", JacksonUtil.obj2Str(callConfig.getCallFeeInfo()));
        }
        if (!ObjectUtils.isEmpty(callConfig.getDoubleCallFeeInfo())) {
            result.put("double_call_fee_info", JacksonUtil.obj2Str(callConfig.getDoubleCallFeeInfo()));
        }
        if (!ObjectUtils.isEmpty(callConfig.getValidDate())) {
            result.put("exten_fee_valid_date", callConfig.getValidDate());
        }
        result.put("way_type", callConfig.getWayType());
        return result;
    }

    private Map<String, String> getDeleteMemberParams(Long userId) {
        Map<String, String> result = new TreeMap<>();
        result.put("user_number", String.valueOf(userId));
        return result;
    }

    private Map<String, String> getForbiddenMemberParams(int status, Long userId) {
        Map<String, String> result = new TreeMap<>();
        result.put("status", String.valueOf(status));
        result.put("user_number", String.valueOf(userId));
        return result;
    }

    private Map<String, String> getAddMemberParams(String phone, String name, Long userId) {
        Map<String, String> result = new TreeMap<>();
        result.put("phone", phone);
        result.put("name", name);
        result.put("user_number", String.valueOf(userId));
        return result;
    }

    private Map<String, String> getEditMemberParams(String newPhone, Long userId) {
        Map<String, String> result = new TreeMap<>();
        result.put("phone", newPhone);
        result.put("user_number", String.valueOf(userId));
        return result;
    }

    private Map<String, String> getExtenInfoParams(Long userId) {
        Map<String, String> map = new TreeMap<>();
        map.put("user_number", String.valueOf(userId));
        return map;
    }


    private Map<String, String> getSignInOrOutParams(String exten, Integer signInType) {
        Map<String, String> result = new TreeMap<>();
        result.put("exten", exten);
        result.put("sign_in", String.valueOf(signInType));
        return result;
    }

    private Map<String, String> getVoiceParams(String url) {
        Map<String, String> result = new TreeMap<>();
        result.put("voice_path", url);
        return result;
    }

    private Map<String, String> getRlExtenListParams() {
        Map<String, String> result = new TreeMap<>();
        result.put("is_login", "1");
        result.put("busy_type", "0");
        return result;
    }


    private Map<String, String> handleVoiceParams(String phone, String content, Integer materialTypeValue, Map<String, String> materialParams) {
        Map<String, String> params = new HashMap<>();
        params.put("tel_b", phone);
        VoiceMaterialTypeEnum materialType = VoiceMaterialTypeEnum.get(materialTypeValue);
        if (VoiceMaterialTypeEnum.TEXT.equals(materialType)) {
            params.put("type", "0");
            // 替换变量
            for (Entry<String, String> param : materialParams.entrySet()) {
                String template = "{" + param.getKey() + "}";
                if (content.contains(template)) {
                    content = content.replace(template, param.getValue());
                }
            }
            String base64Content = Base64.getEncoder().encodeToString(content.getBytes(StandardCharsets.UTF_8));
            params.put("text", base64Content);
        } else if (VoiceMaterialTypeEnum.VOICE.equals(materialType)) {
            params.put("type", "1");
            params.put("file", content);
        }
        return params;
    }
}
