package com.baijiayun.duanxunbao.sdk.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.parser.Feature;
import com.baijiayun.duanxunbao.common.enums.ResultCode;
import com.baijiayun.duanxunbao.common.exception.BusinessException;
import com.baijiayun.duanxunbao.common.utils.HttpClientUtils;
import com.baijiayun.duanxunbao.sdk.service.BjyPartnerService;
import com.baijiayun.duanxunbao.sdk.service.ScrmOpenUrl;
import com.baijiayun.duanxunbao.sdk.utils.UrlUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Service;
import org.springframework.util.DigestUtils;

import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.Map.Entry;

/**
 * @author lanye
 */
@Slf4j
@Service
@RefreshScope
@SuppressWarnings("all")
public class BjyPartnerServiceImpl implements BjyPartnerService {

    @Value("${app.bjy.domain:}")
    private String bjyUrl;

    @Value("${app.bjy.salt:}")
    private String salt;

    @Value("${app.bjy.prefix:}")
    private String prefix;

    @Value("${app.bjy.apiDomain}")
    private String apiDomain;

    @Value("${app.bjy.appKey}")
    private String appKey;

    @Value("${app.bjy.appSecret}")
    private String appSecret;

    private final int TOKEN_CACHE_EXPIRE = 1;

    private final String TIMESTAMP = "timestamp";

    private final String ERROR_MSG = "调用API异常";

    @Override
    public void setSign(Map<String, String> params) {
        for (Entry<String, String> entry : params.entrySet()) {
            if (entry.getValue() == null) {
                continue;
            }
            params.put(entry.getKey(), entry.getValue().trim());
        }
        params.put("sign", getMd5Sign(params));
    }

    @Override
    public JSONObject bjyPost(String url, Map<String, String> map, Map<String, String> header) {
        log.info("bjyPost url:{},params:{}", bjyUrl + url, map);
        Map<String, String> params = putTimestamp(map);
        setSign(params);
        String result;
        String postUrl = UrlUtils.assembleUrl(bjyUrl, url);
        try {
            result = HttpClientUtils.doPost(postUrl, params, header, StandardCharsets.UTF_8);
        } catch (Exception e) {
            log.error("bjy api post error", e);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, ERROR_MSG);
        }
        JSONObject jsonObject = JSON.parseObject(result, Feature.OrderedField);
        int code = jsonObject.getIntValue("code");
        String msg = jsonObject.getString("msg");
        switch (code) {
            case 0: {
                return jsonObject;
            }
            case 30004: {
                log.warn("bjyPost with header warn repeat");
                return jsonObject;
            }
            default:
                log.error("bjyPost with header error,result:{}", result);
                throw new BusinessException(ResultCode.BUSINESS_ERROR, msg);
        }
    }

    @Override
    public JSONObject bjyPost(String partnerId, String token,  String url, Map<String, String> map) {
        log.info("bjyPost url:{},params:{}", bjyUrl + url, map);

        Map<String, String> bjyHeader = new HashMap<>(1);
        bjyHeader.put("Authorization", prefix + token);

        Map<String, String> params = putTimestamp(map);
        map.put("partner_id", partnerId);

        setSign(params);
        String result;
        String postUrl = UrlUtils.assembleUrl(bjyUrl, url);
        try {
            result = HttpClientUtils.doPost(postUrl, params, bjyHeader, StandardCharsets.UTF_8);
        } catch (Exception e) {
            log.error("bjy api post error", e);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, ERROR_MSG);
        }
        JSONObject jsonObject = JSON.parseObject(result, Feature.OrderedField);
        int code = jsonObject.getIntValue("code");
        String msg = jsonObject.getString("msg");
        log.info("bjyPost params:{} ,result:{}", params, result);
        switch (code) {
            case 0: {
                return jsonObject;
            }
            case 30004: {
                log.warn("bjyPost warn repeat");
                return jsonObject;
            }
            case 31002: {
                log.warn("bjyPost voice call phone limie");
                return jsonObject;
            }
            case 31003: {
                log.warn("bjyPost voice call speed limit");
                return jsonObject;
            }
            case 31004: {
                log.warn("bjyPost voice call text over length");
                return jsonObject;
            }
            default:
                log.error("bjyPost error,result:{}", result);
                throw new BusinessException(ResultCode.BUSINESS_ERROR, msg);
        }
    }

    @Override
    public JSONObject bjyPost(String partnerId, String url, Map<String, String> params) {
        log.info("bjyPost partnerId: {}, url:{},params:{}", partnerId, bjyUrl + url, params);

        params = putTimestamp(params);
        params.put("partner_id", partnerId);

        setSign(params);
        String result;
        String postUrl = UrlUtils.assembleUrl(bjyUrl, url);
        try {
            result = HttpClientUtils.doPost(postUrl, params);
        } catch (Exception e) {
            log.error("bjy api post error", e);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, ERROR_MSG);
        }
        JSONObject jsonObject = JSON.parseObject(result, Feature.OrderedField);
        int code = jsonObject.getIntValue("code");
        String msg = jsonObject.getString("msg");
        log.info("bjyPost params:{} ,result:{}", params, result);
        switch (code) {
            case 0: {
                return jsonObject;
            }
            case 30004: {
                log.warn("bjyPost warn repeat");
                return jsonObject;
            }
            case 31002: {
                log.warn("bjyPost voice call phone limie");
                return jsonObject;
            }
            case 31003: {
                log.warn("bjyPost voice call speed limit");
                return jsonObject;
            }
            case 31004: {
                log.warn("bjyPost voice call text over length");
                return jsonObject;
            }
            default:
                log.error("bjyPost error,result:{}", result);
                throw new BusinessException(ResultCode.BUSINESS_ERROR, msg);
        }
    }

    /**
     * sign规则
     * $sign=md5("{$salt}:{$sign_str}")
     * $salt:双方约定 测试环境为baijiayun.com_dev
     * $sign_str:
     * - 发生数据按照键名对关联数组进行升序排序
     * - 数组对应的值按照“$”分割拼接：join("$", $sign_data);
     * - 如果数组value为数组，请先json_encode 在拼接
     */
    private String getMd5Sign(Map<String, String> params) {
        log.info("getMd5Sign params:{}", params);
        StringBuilder sb = new StringBuilder();
        sb.append(salt);
        sb.append(':');

        TreeMap<String, String> sorted = new TreeMap<>();
        sorted.putAll(params);

        for (Entry<String, String> entry : sorted.entrySet()) {
            sb.append(entry.getValue());
            sb.append('$');
        }

        String sign = sb.toString();
        // 移除末尾的 "$"
        sign = sign.substring(0, sign.length() - 1);

        log.info("getMd5Sign sign before:{}", sign);

        sign = DigestUtils.md5DigestAsHex(sign.getBytes());

        log.info("getMd5Sign sign after:{}", sign);

        return sign;
    }

    private String getMd5SignByOpenApi(Map<String, String> params) {
        log.info("getMd5SignByOpenApi params:{}", params);

        // 将请求参数按 key 字典顺序(ASCII 值大小)升序排序。
        TreeMap<String, String> sorted = new TreeMap<>();
        sorted.putAll(params);

        StringBuilder sb = new StringBuilder();
        // 将排好序的参数接成key1=value1&key2=value2&...&keyN=valueN
        for (Entry<String, String> entry : sorted.entrySet()) {
            sb.append(entry.getKey());
            sb.append('=');
            sb.append(entry.getValue());
            sb.append('&');
        }

        // 将以上拼好的串后面再拼上&app_secret=<app_secret>
        sb.append("app_secret=");
        sb.append(appSecret);

        // 对以上拼好的串算一个 32 位 md5 值（小写）
        log.info("getMd5SignByOpenApi sign before:{}", sb.toString());
        String sign = DigestUtils.md5DigestAsHex(sb.toString().getBytes());
        log.info("getMd5SignByOpenApi sign after:{}", sign);

        return sign;
    }

    private Map<String, String> putTimestamp(Map<String, String> map) {
        map.put(TIMESTAMP, String.valueOf(System.currentTimeMillis()));
        return map;
    }

    @Override
    public String getCityByIp(String ip) {
        if (StringUtils.isBlank(ip)) {
            return null;
        }
        Map<String, String> params = Maps.newHashMap();
        params.put("ips", ip);
        try {
            JSONObject result = bjyGet(ScrmOpenUrl.IP_INFO, params);
            JSONObject data = result.getJSONObject("data");
            if (data != null) {
                JSONArray jsonArr = data.getJSONArray(ip);
                if (jsonArr != null) {
                    return jsonArr.getString(2);
                }
            }
        } catch (Exception e) {
            log.error("get city by ip error: ", e);
        }
        return null;
    }

    private JSONObject bjyGet(String url, Map<String, String> map) {
        Map<String, String> params = putTimestamp(map);
        setSign(params);
        log.info("bjyGet url:{},params:{}", bjyUrl + url, params);
        String result;
        try {
            result = HttpClientUtils.doGet(bjyUrl + url, params, StandardCharsets.UTF_8);
        } catch (Exception e) {
            log.error("bjy api get error", e);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, ERROR_MSG);
        }
        JSONObject jsonObject = JSON.parseObject(result, Feature.OrderedField);
        int code = jsonObject.getIntValue("code");
        String msg = jsonObject.getString("msg");
        log.info("bjyGet params:{} ,result:{}", params, result);
        switch (code) {
            case 0: {
                return jsonObject;
            }
            case 30004: {
                log.warn("bjyGet warn repeat");
                return jsonObject;
            }
            default:
                log.error("bjyGet error,result:{}", result);
                throw new BusinessException(ResultCode.BUSINESS_ERROR, msg);
        }
    }

    @Override
    public JSONObject bjyPostByOpenApi(String url, Map<String, String> params) {
        log.info("bjyPostByOpenApi url:{},params:{}", url, params);
        if (params == null) {
            params = Maps.newHashMap();
        }
        params.put("app_key", appKey);
        params.put(TIMESTAMP, String.valueOf(System.currentTimeMillis()));
        // 生成签名
        String sign = getMd5SignByOpenApi(params);
        params.put("sign", sign);

        // 开始请求
        String result;
        try {
            String postUrl = UrlUtils.assembleUrl(apiDomain, url);
            result = HttpClientUtils.doPost(postUrl, params);
        } catch (Exception e) {
            log.error("bjyPostByOpenApi,bjy api post error", e);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, ERROR_MSG);
        }
        JSONObject jsonObject = JSON.parseObject(result, Feature.OrderedField);
        log.info("bjyPostByOpenApi params:{} ,result:{}", params, jsonObject);
        return jsonObject;
    }
}
