package com.baijiayun.duanxunbao.sdk.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baijiayun.duanxunbao.common.enums.ResultCode;
import com.baijiayun.duanxunbao.common.exception.BusinessException;
import com.baijiayun.duanxunbao.sdk.dto.req.*;
import com.baijiayun.duanxunbao.sdk.dto.resp.*;
import com.baijiayun.duanxunbao.sdk.service.AccountService;
import com.baijiayun.duanxunbao.sdk.service.BjyPartnerService;
import com.baijiayun.duanxunbao.sdk.service.ScrmOpenUrl;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.text.MessageFormat;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

@Slf4j
@Service
public class AccountServiceImpl implements AccountService {

    @Autowired
    private BjyPartnerService bjyPartnerService;

    @Value("${app.bjy.privateDomainTemplate}")
    private String privateDomainTemplate;

    @Override
    public PartnerInfoRespDto bindAccount(AccountBindReq reqDto) {
        log.info("bindAccount reqDto: {}", reqDto);
        Map<String, String> params = new TreeMap<>();
        params.put("corp_id", reqDto.getCorpId());
        params.put("corp_name", reqDto.getCorpName());
        params.put("phone", reqDto.getPhone());

        JSONObject account = bjyPartnerService.bjyPost(ScrmOpenUrl.ACCOUNT_BIND, params, Maps.newHashMap()).getJSONObject("data");
        PartnerInfoRespDto respDto = new PartnerInfoRespDto();
        respDto.setPartnerId(account.getString("partner_id"));
        respDto.setPartnerKey(account.getString("partner_key"));
        respDto.setSecretKey(account.getString("secret_key"));
        respDto.setEmail(account.getString("email"));
        respDto.setMobile(account.getString("mobile"));
        respDto.setPartnerName(account.getString("partner_name"));
        respDto.setPrivateDomainUrl(account.getString("private_domain_url"));
        return respDto;
    }

    @Override
    public PartnerInfoRespDto isPhoneExist(String phone) {
        log.info("isPhoneExist phone:{}", phone);
        Preconditions.checkArgument(StringUtils.isNotBlank(phone), "phone is blank");

        Map<String, String> params = new TreeMap<>();
        params.put("phone", phone);

        JSONObject jsonObject = bjyPartnerService.bjyPost(ScrmOpenUrl.ACCOUNT_PHONE_EXIST, params, Maps.newHashMap());
        log.info("isPhoneExist get jsonObject: {}", jsonObject);

        PartnerInfoRespDto partnerInfoRespDto = new PartnerInfoRespDto();

        Integer exist = jsonObject.getJSONObject("data").getInteger("exist");
        Integer bind = jsonObject.getJSONObject("data").getInteger("bind");
        String mobile = jsonObject.getJSONObject("data").getString("phone");
        JSONObject partnerObj = jsonObject.getJSONObject("data").getJSONObject("partner");
        if (partnerObj != null) {
            String partnerId = partnerObj.getString("partner_id");
            String partnerKey = partnerObj.getString("partner_key");
            String secretKey = partnerObj.getString("secret_key");
            String email = partnerObj.getString("email");
            String partnerName = partnerObj.getString("partner_name");
            String domainUrl = partnerObj.getString("private_domain");

            partnerInfoRespDto.setPartnerId(partnerId);
            partnerInfoRespDto.setPartnerKey(partnerKey);
            partnerInfoRespDto.setSecretKey(secretKey);
            partnerInfoRespDto.setEmail(email);
            partnerInfoRespDto.setPartnerName(partnerName);
            partnerInfoRespDto.setPrivateDomainUrl(domainUrl);
        }
        partnerInfoRespDto.setExist(exist);
        partnerInfoRespDto.setBind(bind);
        partnerInfoRespDto.setMobile(mobile);
        return partnerInfoRespDto;
    }

    @Override
    public PartnerInfoRespDto getPrivateInfo(String partnerId) {
        log.info("getPartnerInfo partnerId:{}", partnerId);
        Preconditions.checkArgument(StringUtils.isNotBlank(partnerId), "partnerId is blank");

        Map<String, String> params = new TreeMap<>();
        params.put("partner_id", partnerId);

        JSONObject jsonObject = bjyPartnerService.bjyPost(ScrmOpenUrl.ACCOUNT_PRIVATE_INFO, params, Maps.newHashMap());
        log.info("getPartnerInfo jsonObject: {}", jsonObject);

        PartnerInfoRespDto partnerInfoRespDto = new PartnerInfoRespDto();

        Integer exist = jsonObject.getJSONObject("data").getInteger("exist");
        JSONObject partnerObj = jsonObject.getJSONObject("data").getJSONObject("partner");
        if (partnerObj != null) {
            String partnerKey = partnerObj.getString("partner_key");
            String secretKey = partnerObj.getString("secret_key");
            String email = partnerObj.getString("email");
            String partnerName = partnerObj.getString("partner_name");
            String domainUrl = partnerObj.getString("private_domain");
            String mobile = partnerObj.getString("phone");

            partnerInfoRespDto.setPartnerId(partnerId);
            partnerInfoRespDto.setPartnerKey(partnerKey);
            partnerInfoRespDto.setSecretKey(secretKey);
            partnerInfoRespDto.setEmail(email);
            partnerInfoRespDto.setPartnerName(partnerName);
            partnerInfoRespDto.setPrivateDomainUrl(domainUrl);
            partnerInfoRespDto.setMobile(mobile);
        }
        return partnerInfoRespDto;
    }

    @Override
    public int getAccountRestMoney(AccountBaseReq req) {
        log.info("getAccountRestMoney, req:{}", req);
        Map<String, String> queryParams = new TreeMap<>();
        queryParams.put("partner_id", req.getPartnerId());
        queryParams.put("timestamp", String.valueOf(System.currentTimeMillis()));

        JSONObject jsonObject;
        try {
            jsonObject = bjyPartnerService.bjyPost(req.getPartnerId(), req.getToken(), ScrmOpenUrl.ACCOUNT_GET_BALANCE, queryParams);
            log.info("get restMoney result:{}", jsonObject);
        } catch (Exception ex) {
            log.error("调用获取商户余额接口失败:", ex);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "获取商户可用余额失败,请稍后重试");
        }

        int code = jsonObject.getIntValue("code");
        if (code != 0) {
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "获取商户可用余额失败,请稍后重试");
        }
        return jsonObject.getJSONObject("data").getIntValue("left_total_money");
    }

    /**
     * 开通百家云账号
     *
     */
    @Override
    public PartnerInfoRespDto createScrmAccount(String corpId, String corpName, String mobile) {
        Map<String, String> params = new TreeMap<>();
        params.put("corp_id", corpId);
        params.put("corp_name", corpName);
        params.put("phone", mobile);
        params.put("timestamp", String.valueOf(System.currentTimeMillis()));

        JSONObject account = bjyPartnerService.bjyPost(ScrmOpenUrl.CREATE_ACCOUNT, params, Maps.newHashMap()).getJSONObject("data");
        if (ObjectUtils.isEmpty(account)) {
            return null;
        }
        PartnerInfoRespDto respDto = new PartnerInfoRespDto();
        respDto.setPartnerId(account.getString("partner_id"));
        respDto.setPartnerKey(account.getString("partner_key"));
        respDto.setSecretKey(account.getString("secret_key"));
        respDto.setEmail(account.getString("email"));
        respDto.setMobile(account.getString("mobile"));
        respDto.setPartnerName(account.getString("partner_name"));
        respDto.setPrivateDomainUrl(account.getString("private_domain_url"));
        return respDto;
    }

    @Override
    public String scrmLogin(String corpId) {
        Map<String, String> params = new TreeMap<>();
        params.put("corp_id", corpId);
        params.put("timestamp", String.valueOf(System.currentTimeMillis()));

        String token;
        try {
            JSONObject result = bjyPartnerService.bjyPost(ScrmOpenUrl.LOGIN_URL, params, Maps.newHashMap());
            log.info("getBjyToken params:{},result:{}", params, result);
            token = result.getJSONObject("data").getString("token");
            return token;
        } catch (Exception e) {
            log.error("getBjyToken error", e);
            token = StringUtils.EMPTY;
        }
        return token;
    }

    @Override
    public String dxbLogin(String partnerId) {
        Map<String, String> params = new TreeMap<>();
        params.put("partner_id", partnerId);
        params.put("app_from", "dxb");
        params.put("timestamp", String.valueOf(System.currentTimeMillis()));
        log.info("dxbLogin get params: {}", params);
        String token;
        try {
            JSONObject result = bjyPartnerService.bjyPost(ScrmOpenUrl.LOGIN_URL_DXB, params, Maps.newHashMap());
            log.info("dxbLogin params:{},result:{}", params, result);
            token = result.getJSONObject("data").getString("token");
            return token;
        } catch (Exception e) {
            log.error("dxbLogin error", e);
            token = StringUtils.EMPTY;
        }
        return token;
    }


    @Override
    public String scrmAccountConfig(AccountConfigDto dto) {
        log.info("scrmAccountConfig dto: {}", dto);
        Map<String, String> param = Maps.newHashMap();
        param.put("live_chat_keywork_callback_url", dto.getKeywordCallBackUrl());
        param.put("enable_live_chat_keyword", dto.getEnableLiveKeyword());
        param.put("enable_user_event_callback", dto.getEnableUserEvent());
        param.put("user_event_callback_url", dto.getUserEventCallBackUrl());
        param.put("enable_live_sell_pc_student", dto.getEnableLiveSellPc());
        param.put("enable_use_lucky_bag", dto.getEnableUserLuckyBug());

        JSONObject response = bjyPartnerService.bjyPost(dto.getPartnerId(), dto.getToken(), ScrmOpenUrl.SCRM_ACCOUNT_CONFIG, param);
        Map<String, String> result = Maps.newHashMap();
        result.put("param", JSON.toJSONString(param));
        result.put("response", JSON.toJSONString(response));
        result.put("url", ScrmOpenUrl.SCRM_ACCOUNT_CONFIG);
        return JSON.toJSONString(result);
    }



    @Override
    public String initAccount(InitAccountReqDto reqDto) {
        Map<String, String> param = Maps.newHashMap();
        param.put("user_event_callback_url", reqDto.getUserEventCallbackUrl());
        param.put("live_award_callback_url", reqDto.getLiveAwardCallbackUrl());
        param.put("live_chat_keyword_callback_url", reqDto.getLiveChatKeyworkCallbackUrl());
        param.put("live_room_callback_url", reqDto.getLiveRoomCallbackUrl());
        param.put("chat_message_callback_url", reqDto.getChatMessageCallbackUrl());
        param.put("live_operator_callback_url", reqDto.getLiveOperatorCallbackUrl());
        param.put("live_room_signal_push_callback_url", reqDto.getLiveSignalCallBackUrl());
        param.put("transcode_callback_url", reqDto.getLiveTranscodeCallbackUrl());
        log.info("initAccount param: {}", param);

        JSONObject response = bjyPartnerService.bjyPost(reqDto.getPartnerId(), reqDto.getToken(), ScrmOpenUrl.INIT_ACCOUNT, param);
        Map<String, String> result = Maps.newHashMap();
        result.put("param", JSON.toJSONString(param));
        result.put("response", JSON.toJSONString(response));
        result.put("url", ScrmOpenUrl.INIT_ACCOUNT);
        return JSON.toJSONString(result);
    }

    @Override
    public AccountProductResp accountProduct(AccountBaseReq req) {
        log.info("accountProduct reqDto: {}", req);
        Map<String, String> queryParams = Maps.newHashMap();

        JSONObject jsonObject;
        try {
            jsonObject = bjyPartnerService.bjyPost(req.getPartnerId(), req.getToken(), ScrmOpenUrl.ACCOUNT_PRODUCT, queryParams);
            log.info("accountProduct result:{}", jsonObject);
        } catch (Exception ex) {
            log.error("accountProduct 调用接口失败:", ex);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "accountProduct 调用接口失败,请稍后重试");
        }
        int code = jsonObject.getIntValue("code");
        if (code != 0) {
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "accountProduct 调用接口失败,请稍后重试");
        }
        JSONArray product =  jsonObject.getJSONObject("data").getJSONArray("product");
        List<Integer> productList = product.toJavaList(Integer.class);
        log.info("accountProduct product: {}, productList: {}", product, productList);
        AccountProductResp resp = new AccountProductResp();
        resp.setProduct(productList);
        return resp;
    }

    @Override
    public AccountProductResp openAccountProduct(AccountProductOpenReq req) {
        log.info("openAccountProduct reqDto: {}", req);
        Map<String, String> openParams = Maps.newHashMap();
        openParams.put("product_type", String.valueOf(req.getProductType()));

        JSONObject jsonObject;
        try {
            jsonObject = bjyPartnerService.bjyPost(req.getPartnerId(), req.getToken(), ScrmOpenUrl.ACCOUNT_PRODUCT_OPEN, openParams);
            log.info("openAccountProduct result:{}", jsonObject);
        } catch (Exception ex) {
            log.error("openAccountProduct 调用接口失败:", ex);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "openAccountProduct 调用接口失败,请稍后重试");
        }
        int code = jsonObject.getIntValue("code");
        if (code != 0) {
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "openAccountProduct 调用接口失败,请稍后重试");
        }
        Boolean isOpen = jsonObject.getBoolean("data");
        log.info("openAccountProduct isOpen: {}", isOpen);
        AccountProductResp resp = new AccountProductResp();
        resp.setIsOpen(isOpen);
        return resp;
    }

    @Override
    public TicketInfoResp getTicketInfo(String ticket) {
        log.info("getTicketInfo ticket: {}", ticket);
        Map<String, String> params = Maps.newHashMap();
        params.put("ticket", ticket);

        // 请求数据
        JSONObject jsonObject;
        try {
            jsonObject = bjyPartnerService.bjyPostByOpenApi(ScrmOpenUrl.GET_TICKET_INFO, params);
            log.info("getTicketInfo result:{}", jsonObject);
        } catch (Exception ex) {
            log.error("getTicketInfo 调用接口失败:", ex);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "getTicketInfo 调用接口失败,请稍后重试");
        }
        int code = jsonObject.getIntValue("code");
        if (code != 0) {
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "getTicketInfo 调用接口失败,请稍后重试");
        }
        JSONObject data = jsonObject.getJSONObject("data");
        if (ObjectUtils.isEmpty(data)) {
            return null;
        }
        TicketInfoResp ticketInfoResp = JSONObject.parseObject(data.toJSONString(), TicketInfoResp.class);
        if (ticketInfoResp != null && StringUtils.isNotBlank(ticketInfoResp.getPrivateDomain())) {
            // 构造完整的私有域名
            ticketInfoResp.setFullPrivateDomain(MessageFormat.format(privateDomainTemplate, ticketInfoResp.getPrivateDomain()));
        }
        return ticketInfoResp;
    }

    @Override
    public LiveInfoResp queryRoomInfo(AccountLiveInfoReq req) {
        log.info("queryRoomInfo req: {}", req);

        Map<String, String> params = Maps.newHashMap();
        params.put("room_id", String.valueOf(req.getRoomId()));

        JSONObject jsonObject;
        try {
            jsonObject = bjyPartnerService.bjyPost(req.getPartnerId(), req.getToken(), ScrmOpenUrl.LIVE_ROOM_INFO, params);
            log.info("queryRoomInfo result:{}", jsonObject);
        } catch (Exception ex) {
            log.error("queryRoomInfo 调用接口失败:", ex);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "openAccountProduct 调用接口失败,请稍后重试");
        }
        int code = jsonObject.getIntValue("code");
        if (code != 0) {
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "openAccountProduct 调用接口失败,请稍后重试");
        }
        JSONObject data = jsonObject.getJSONObject("data");
        if (ObjectUtils.isEmpty(data)) {
            return null;
        }
        return JSONObject.parseObject(data.toJSONString(), LiveInfoResp.class);
    }

    @Override
    public JSONObject getAccountConfig(AccountGetConfigReq req) {
        log.info("getAccountConfig req: {}", req);

        Map<String, String> params = Maps.newHashMap();
        params.put("keys", String.valueOf(req.getKey()));
        params.put("product_type", String.valueOf(req.getProductType()));

        JSONObject jsonObject;
        try {
            jsonObject = bjyPartnerService.bjyPost(req.getPartnerId(), req.getToken(), ScrmOpenUrl.ACCOUNT_GET_CONFIG, params);
            log.info("getAccountConfig result:{}", jsonObject);
        } catch (Exception ex) {
            log.error("getAccountConfig 调用接口失败:", ex);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "openAccountProduct 调用接口失败,请稍后重试");
        }
        int code = jsonObject.getIntValue("code");
        String msg = jsonObject.getString("msg");
        if (code != 0) {
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "openAccountProduct 调用接口失败:" + msg);
        }
        JSONObject data = jsonObject.getJSONObject("data");
        if (ObjectUtils.isEmpty(data)) {
            return null;
        }
        return data;
    }

    @Override
    public void updateAccountConfig(AccountUpdateConfigReq req) {
        log.info("updateAccountConfig req: {}", req);

        Map<String, String> params = Maps.newHashMap();
        params.put("key", String.valueOf(req.getKey()));
        params.put("product_type", String.valueOf(req.getProductType()));
        params.put("value", String.valueOf(req.getValue()));

        JSONObject jsonObject;
        try {
            jsonObject = bjyPartnerService.bjyPost(req.getPartnerId(), req.getToken(), ScrmOpenUrl.ACCOUNT_UPDATE_CONFIG, params);
            log.info("updateAccountConfig result:{}", jsonObject);
        } catch (Exception ex) {
            log.error("updateAccountConfig 调用接口失败:", ex);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "updateAccountConfig 调用接口失败,请稍后重试");
        }
        int code = jsonObject.getIntValue("code");
        String msg = jsonObject.getString("msg");
        if (code != 0) {
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "openAccountProduct 调用接口失败:" + msg);
        }
    }

    @Override
    public AccountInfoResp getAccountInfo(AccountBaseReq req) {
        log.info("getAccountInfo req: {}", req);

        Map<String, String> queryParams = Maps.newHashMap();
        JSONObject jsonObject;
        try {
            jsonObject = bjyPartnerService.bjyPost(req.getPartnerId(), req.getToken(), ScrmOpenUrl.SCRM_ACCOUNT_GETINFO, queryParams);
            log.info("getAccountInfo result:{}", jsonObject);
        } catch (Exception ex) {
            log.error("getAccountInfo 调用接口失败:", ex);
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "getAccountInfo 调用接口失败,请稍后重试");
        }
        int code = jsonObject.getIntValue("code");
        String msg = jsonObject.getString("msg");
        if (code != 0) {
            throw new BusinessException(ResultCode.BUSINESS_ERROR, "getAccountInfo 调用接口失败:" + msg);
        }
        JSONObject data = jsonObject.getJSONObject("data");
        if (ObjectUtils.isEmpty(data)) {
            return null;
        }
        return JSONObject.parseObject(data.toJSONString(), AccountInfoResp.class);
    }
}
