/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.entity.message.resp;

import cn.kinyun.scrm.weixin.sdk.enums.WxMsgType;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import cn.kinyun.scrm.weixin.sdk.utils.MsgXmlBuilder;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NonNull;
import lombok.ToString;

/**
 * 回复文本消息
 * 
 * @title TextMsg
 * @desc 回复文本消息
 * @author yanmaoyuan
 * @date 2019年4月25日
 * @version 1.0
 */
@Data
@EqualsAndHashCode(callSuper = true)
@ToString(callSuper = true)
public class TextMsg extends BaseRespMsg {

    private static final long serialVersionUID = 2896171843831253400L;

    /**
     * 用于构造文本内容
     */
    @JsonIgnore
    private StringBuffer contentBuilder = new StringBuffer();

    public TextMsg() {
        this.msgType = WxMsgType.Text.getValue();
    }
    
    /**
     * 回复的消息内容（换行：在content中能够换行，微信客户端就支持换行显示）
     */
    @JsonProperty("content")
    public String getContent() {
        return contentBuilder.toString();
    }

    /**
     * 设置回复的消息内容
     * 
     * @param content
     */
    public void setContent(String content) {
        contentBuilder = new StringBuffer(content);
    }

    /**
     * 向文本消息中追加一个字符串
     * 
     * @param content
     */
    public TextMsg print(String content) {
        if (content != null) {
            contentBuilder.append(content);
        }
        return this;
    }

    /**
     * 向文本消息中追加一个字符串，末尾自动换行。
     * 
     * @param content
     */
    public TextMsg println(String content) {
        if (content != null) {
            contentBuilder.append(content);
        }
        contentBuilder.append('\n');
        return this;
    }

    /**
     * 向文本消息中追加一个换行符。
     */
    public TextMsg println() {
        contentBuilder.append('\n');
        return this;
    }

    /**
     * 向文本消息中追加一行字符串，使用格式化输出。
     * 
     * @param format
     * @param args
     */
    public TextMsg printf(String format, Object...args) {
        String content = String.format(format, args);
        contentBuilder.append(content);
        return this;
    }

    /**
     * 添加超链接
     * 
     * @param title 标题
     * @param url 链接地址
     * @return
     */
    public TextMsg addHyperlink(@NonNull String title, @NonNull String url) {
        contentBuilder.append("<a href=\"").append(url).append("\">");
        contentBuilder.append(title);
        contentBuilder.append("</a>");
        return this;
    }

    /**
     * 添加小程序
     * 
     * @param title 标题
     * @param appId 小程序ID
     * @param appPath 小程序页面路径
     * @param url 链接地址
     * @return
     */
    public TextMsg addMiniProgram(@NonNull String title, @NonNull String appId, @NonNull String appPath, String url) {
        contentBuilder.append("<a");
        contentBuilder.append(" href=\"").append(url).append('\"');
        contentBuilder.append(" data-miniprogram-appid=\"").append(appId).append('\"');
        contentBuilder.append(" data-miniprogram-path=\"").append(appPath).append('\"');
        contentBuilder.append('>');
        contentBuilder.append(title);
        contentBuilder.append("</a>");
        return this;
    }

    @Override
    public String toXml() {
        MsgXmlBuilder builder = new MsgXmlBuilder(super.toXml());

        builder.addCData("Content", contentBuilder.toString());
        builder.surroundWith("xml");

        return builder.toString();
    }
}