/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api.custom;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.custom.CustomAccount;
import cn.kinyun.scrm.weixin.sdk.entity.custom.CustomAccountList;
import cn.kinyun.scrm.weixin.sdk.entity.custom.OnlineCustomAccount;
import cn.kinyun.scrm.weixin.sdk.entity.custom.OnlineList;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.List;
import java.util.Map;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 微信客服账号接口
 * 
 * @title WxCustomAccountAPI
 * @desc 微信客服账号接口
 * @author yanmaoyuan
 * @date 2019年4月28日
 * @version 1.0
 * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1421140547">客服消息</a>
 * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1458044813">新客服管理</a>
 */
@Slf4j
@Component
public class WxCustomAccountAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 添加客服账号 POST
     */
    @Value("${wx.custom.kfaccount.add}")
    private String wxCustomAccountAdd;

    /**
     * 修改客服账号 POST
     */
    @Value("${wx.custom.kfaccount.update")
    private String wxCustomAccountUpdate;

    /**
     * 删除客服账号 GET
     */
    @Value("${wx.custom.kfaccount.del")
    private String wxCustomAccountDelete;

    /**
     * 邀请绑定客服帐号 POST
     */
    @Value("${wx.custom.kfaccount.inviteworker")
    private String wxCustomAccountInviteWorker;

    /**
     * 设置客服帐号的头像 POST/FORM
     */
    @Value("${wx.custom.kfaccount.uploadheadimg")
    private String wxCustomAccountUploadHeadImg;

    /**
     * 获得客服账号列表 GET
     */
    @Value("${wx.custom.kfaccount.getkflist")
    private String wxCustomAccountGetKflist;

    /**
     * 获得在线客服列表 GET
     */
    @Value("${wx.custom.kfaccount.getkflist")
    private String wxCustomAccountGetOnlineKflist;

    /**
     * 添加客服帐号.
     * 
     * 发者可以通过本接口为公众号添加客服账号，每个公众号最多添加10个客服账号。
     * 
     * @param accessToken 接口调用凭证
     * @param params 客服账号. 其中account, nickname 字段为必填
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public void add(@NonNull String accessToken, @NonNull CustomAccount params) throws WeixinException {
        log.info("add custom account with params={}", params);
        Preconditions.checkArgument(StringUtils.isNoneBlank(params.getAccount()), "客服账号名为空");
        Preconditions.checkArgument(StringUtils.isNoneBlank(params.getNickname()), "客服昵称为空");

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        HttpEntity<CustomAccount> request = new HttpEntity<CustomAccount>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxCustomAccountAdd, accessToken);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        WeixinException.isSuccess(response.getBody());// 处理错误码
    }

    /**
     * 修改客服帐号.
     * 
     * 开发者可以通过本接口为公众号修改客服账号。
     * 
     * @param accessToken 接口调用凭证
     * @param params 客服账号. 其中account, nickname 字段为必填
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public void update(@NonNull String accessToken, @NonNull CustomAccount params) throws WeixinException {
        log.info("update custom account with params={}", params);
        Preconditions.checkArgument(StringUtils.isNoneBlank(params.getAccount()), "客服账号名为空");
        Preconditions.checkArgument(StringUtils.isNoneBlank(params.getNickname()), "客服昵称为空");

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        HttpEntity<CustomAccount> request = new HttpEntity<CustomAccount>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxCustomAccountUpdate, accessToken);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        WeixinException.isSuccess(response.getBody());// 处理错误码
    }

    /**
     * 删除客服账号.
     * 
     * 开发者可以通过该接口为公众号删除客服帐号。
     * 
     * @param accessToken 接口调用凭证
     * @param kfAccount 客服账号
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public void delete(@NonNull String accessToken, @NonNull String kfAccount) throws WeixinException {
        log.info("delete custom account with kf_account={}", kfAccount);
        Preconditions.checkArgument(StringUtils.isNoneBlank(kfAccount), "客服账号名为空");

        // 发送请求
        String url = MessageFormat.format(wxCustomAccountDelete, accessToken, kfAccount);
        ResponseEntity<ErrorCode> response = restTemplate.getForEntity(url, ErrorCode.class);

        WeixinException.isSuccess(response.getBody());// 处理错误码
    }

    /**
     * 邀请绑定客服帐号
     * 
     * @param accessToken 接口调用凭证
     * @param kfAccount 客服账号
     * @param inviteWx 接收绑定邀请的客服微信号
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public void inviteWorker(@NonNull String accessToken, @NonNull String kfAccount, @NonNull String inviteWx)
        throws WeixinException {
        log.info("bind kf_account={} with invite_wx={}", kfAccount, inviteWx);
        // 发送请求
        Preconditions.checkArgument(StringUtils.isNoneBlank(kfAccount), "客服账号名为空");
        Preconditions.checkArgument(StringUtils.isNoneBlank(inviteWx), "受邀客服微信号为空");

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        Map<String, Object> params = Maps.newHashMap();
        params.put("kf_account", kfAccount);
        params.put("invite_wx", inviteWx);
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxCustomAccountInviteWorker, accessToken);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        WeixinException.isSuccess(response.getBody());// 处理错误码
    }

    /**
     * 设置客服帐号的头像.
     * 
     * 开发者可调用本接口来上传图片作为客服人员的头像，头像图片文件必须是jpg格式，推荐使用640*640大小的图片以达到最佳效果。
     * 
     * @param accessToken
     * @param account 客服账号
     * @param media 该参数仅在设置客服头像时出现，是form-data中媒体文件标识，有filename、filelength、content-type等信息
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public void uploadHeadImg(@NonNull String accessToken, @NonNull String account, @NonNull Resource media)
        throws WeixinException {
        log.info("upload head img with account={}, media={}", account, media);
        Preconditions.checkArgument(StringUtils.isNoneBlank(account), "客服账号名为空");
        Preconditions.checkArgument(media != null, "头像文件为空");
        Preconditions.checkArgument(media.getFilename().toLowerCase().endsWith("jpg"), "头像图片文件必须是jpg格式");

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.MULTIPART_FORM_DATA);

        // 构造请求体
        Map<String, Object> params = Maps.newHashMap();
        params.put("media", media);

        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxCustomAccountUploadHeadImg, accessToken, account);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        WeixinException.isSuccess(response.getBody());// 处理错误码
    }

    /**
     * 获取所有客服账号.
     * 
     * 开发者通过本接口，获取公众号中所设置的客服基本信息，包括客服工号、客服昵称、客服登录账号。
     * 
     * @param accessToken 接口调用凭证
     * @return 客服账号列表
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public List<CustomAccount> getAll(@NonNull String accessToken) throws WeixinException {
        log.info("get custom account list");

        // 发送请求
        String url = MessageFormat.format(wxCustomAccountGetKflist, accessToken);
        ResponseEntity<CustomAccountList> response = restTemplate.getForEntity(url, CustomAccountList.class);

        CustomAccountList result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码

        return result.getList();
    }

    /**
     * 获取在线客服列表
     * 
     * @param accessToken 接口调用凭证
     * @return 在线客服列表
     * @throws WeixinException
     */
    public List<OnlineCustomAccount> getOnlineList(@NonNull String accessToken) throws WeixinException {
        log.info("get online custom account list");

        // 发送请求
        String url = MessageFormat.format(wxCustomAccountGetOnlineKflist, accessToken);
        ResponseEntity<OnlineList> response = restTemplate.getForEntity(url, OnlineList.class);

        OnlineList result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码

        return result.getList();
    }

}