package com.baijiayun.duanxunbao.common.dto;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

/**
 * 节点类
 * 
 * @author shanyu
 *
 */
@Data
public class Node implements Serializable {
    private static final long serialVersionUID = 7052759963336265589L;

    public static final int NAME_SIZE = 3;
    public static final String SEPARATE = "-";
    public static final int DEEPEST_LEVEL = 10;
    public static final String SEPARATOR = ",";

    private Long bizId;

    private long id;

    private List<Long> ids;

    private Long pid;

    private int level;

    private Long type;// 部门类型

    private String num;// 唯一编码

    private String code;

    private String name;

    private List<String> names;

    private List<Node> children;

    private Set<Long> posterityIds;

    private int isDel;

    private String fullName;
    
    /**
     * 部门唯一编码
     */
    private String sysCode;

    public String getFullName() {
        fullName = getFullName(this);
        return fullName;
    }

    /**
     * 获取节点信息，不带子节点
     * 
     * @param nodeId
     * @param nodeMap
     * @return
     */
    public static Node getNodeWithoutChildren(Long nodeId, Map<Long, Node> nodeMap) {
        Node node = nodeMap.get(nodeId);
        if (node != null) {
            Node result = new Node();
            result.setId(node.getId());
            result.setNum(node.getNum());
            result.setCode(node.getCode());
            result.setType(node.getType());
            result.setIds(node.getIds());
            result.setLevel(node.getLevel());
            result.setName(node.getName());
            result.setNames(node.getNames());
            result.setPid(node.getPid());
            result.setBizId(node.getBizId());
            return result;
        }
        return null;
    }

    /**
     * 获取组织架构节点名
     * 
     * @return
     */
    public static String getFullName(Node node) {
        if (CollectionUtils.isEmpty(node.getNames())) {
            return StringUtils.EMPTY;
        } else {
            int size = node.getNames().size();
            if (node.getNames().size() > NAME_SIZE) {
                StringBuilder builder = new StringBuilder();
                builder.append(node.getNames().get(size - 3)).append(SEPARATE).append(node.getNames().get(size - 2))
                    .append(SEPARATE).append(node.getNames().get(size - 1));
                return builder.toString();
            } else {
                return StringUtils.join(node.getNames(), SEPARATE);
            }
        }
    }

    public static Map<Long, Node> buildNodeMap(List<Node> nodes) {
        if (CollectionUtils.isEmpty(nodes)) {
            return Maps.newHashMap();
        }
        Set<Long> rootIds = Sets.newHashSet();
        Map<Long, Node> map = nodes.stream().collect(Collectors.toMap(Node::getId, Function.identity()));
        for (Node node : nodes) {
            if (map.containsKey(node.getPid())) {// 有父节点
                Node parent = map.get(node.getPid());
                if (parent.getChildren() == null) {
                    parent.setChildren(Lists.newArrayList());
                }
                parent.getChildren().add(node);
            } else {
                rootIds.add(node.getId());// 一级节点
            }
        }

        buildIdAndNames(map, rootIds);
        return map;
    }

    private static void buildIdAndNames(Map<Long, Node> nodeMap, Set<Long> rootIds) {
        Set<Long> dealedIds = Sets.newHashSet();// 已处理
        List<Long> tmpList = Lists.newArrayList();// 待处理
        for (Long rootId : rootIds) {
            Node node = nodeMap.get(rootId);
            node.setIds(Lists.newArrayList(node.getId()));
            node.setNames(Lists.newArrayList(node.getName()));
            dealedIds.add(rootId);
            if (CollectionUtils.isNotEmpty(node.getChildren())) {
                for (Node children : node.getChildren()) {
                    if (!dealedIds.contains(children.getId()) && !tmpList.contains(children.getId())) {
                        tmpList.add(children.getId());
                    }
                }
            }
        }
        while (CollectionUtils.isNotEmpty(tmpList)) {
            Long nodeId = tmpList.remove(0);
            Node node = nodeMap.get(nodeId);
            Node pNode = nodeMap.get(node.getPid());
            node.setIds(Lists.newArrayList());
            node.setNames(Lists.newArrayList());
            node.getIds().addAll(pNode.getIds());
            node.getIds().add(node.getId());
            node.getNames().addAll(pNode.getNames());
            node.getNames().add(node.getName());
            dealedIds.add(nodeId);
            if (CollectionUtils.isNotEmpty(node.getChildren())) {
                for (Node children : node.getChildren()) {
                    if (!dealedIds.contains(children.getId()) && !tmpList.contains(children.getId())) {
                        tmpList.add(children.getId());
                    }
                }
            }
        }

    }

    private static void buildIdAndNames(Node node, List<Long> parentIds, List<String> parentNames) {
        List<Long> ids = Lists.newArrayList();
        List<String> names = Lists.newArrayList();
        node.setIds(ids);
        node.setNames(names);
        if (parentIds != null) {
            ids.addAll(parentIds);
            names.addAll(parentNames);
        }
        ids.add(node.getId());
        names.add(node.getName());
        if (CollectionUtils.isNotEmpty(node.getChildren())) {
            for (Node child : node.getChildren()) {
                buildIdAndNames(child, ids, names);
            }
        }
    }

    /**
     * 构造节点树
     * 
     * @param nodes
     * @return
     */
    public static List<Node> buildNodeTrees(List<Node> nodes) {
        List<Node> nodeTrees = Lists.newArrayList();
        if (CollectionUtils.isEmpty(nodes)) {
            return nodeTrees;
        }
        Map<Long, Node> map = nodes.stream().collect(Collectors.toMap(Node::getId, Function.identity()));
        for (Node node : nodes) {
            if (!map.containsKey(node.getPid())) {// 没有父节点
                nodeTrees.add(node);
            } else {
                Node parent = map.get(node.getPid());
                if (parent.getChildren() == null) {
                    parent.setChildren(Lists.newArrayList());
                }
                parent.getChildren().add(node);
            }
        }
        for (Node node : nodes) {
            if (!map.containsKey(node.getPid())) {// 没有父节点
                buildIdAndNames(node, null, null);
            }
        }
        return nodeTrees;
    }

    /**
     * 获取节点及后代节点信息
     * 
     * @param nodeId
     * @param nodeMap
     * @return
     */
    public static Set<Long> getPosterityIds(Long nodeId, Map<Long, Node> nodeMap) {
        Set<Long> nodeIds = Sets.newHashSet();
        nodeIds.add(nodeId);
        Node node = nodeMap.get(nodeId);
        if (node != null) {
            nodeIds.add(nodeId);
            if (CollectionUtils.isNotEmpty(node.getChildren())) {
                for (Node child : node.getChildren()) {
                    getPosterityIds(nodeIds, child.getId(), nodeMap, 1);
                }
            }
        }
        return nodeIds;
    }

    private static void getPosterityIds(Set<Long> nodeIds, Long nodeId, Map<Long, Node> nodeMap, int level) {
        Node node = nodeMap.get(nodeId);
        if (node != null) {
            nodeIds.add(nodeId);
            if (level == DEEPEST_LEVEL) {
                return;
            }
            if (CollectionUtils.isNotEmpty(node.getChildren())) {
                for (Node child : node.getChildren()) {
                    getPosterityIds(nodeIds, child.getId(), nodeMap, level + 1);
                }
            }
        }
    }
}
