package com.kuaike.skynet.logic.wechat.utils;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.kuaike.common.entity.WechatMessage;
import com.kuaike.skynet.logic.wechat.msg.JoinGroupMsg;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

@Slf4j
public class MsgResolver {

    private static final XmlMapper xmlMapper;

    static {
        xmlMapper = new XmlMapper();
        xmlMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    }

    public static Map<String, Object> getMainContent(WechatMessage wechatMessage) {
        return getMainContent(wechatMessage.getMessage());
    }

    public static Map<String, Object> getMainContent(String message) {
        Map<String, Object> map = Maps.newHashMap();
        try {
            message = getInvalidMsg(message);
            JoinGroupMsg joinGroupMessage = xmlMapper.readValue(message, JoinGroupMsg.class);
            JoinGroupMsg.ContentTemplate contentTemplate = joinGroupMessage.getSysMsgTemplate().getContentTemplate();
            String template = contentTemplate.getTemplate();
            List<JoinGroupMsg.Link> linklist = contentTemplate.getLinklist();

            Map<String, String> contentMap = getReplacedContent(linklist);
            if (Objects.nonNull(contentMap)) {
                Set<String> placeholderNames = contentMap.keySet();
                for (String name : placeholderNames) {
                    log.info("placeholder={}", name);
                    template = template.replace("$" + name + "$", contentMap.get(name));
                }
            }
            map.put("content", template);
            setRevokeInfo(linklist, map);
        } catch (Exception e) {
            log.error("解析入群消息失败:{}", message, e);
            map.put("content", "[这是一条系统消息]");
        }
        return map;
    }

    private static void setRevokeInfo(List<JoinGroupMsg.Link> linklist, Map<String, Object> map) {
        for (JoinGroupMsg.Link link : linklist) {
            String name = link.getName();
            if ("revoke".equals(name)) {
                List<String> wechatIds = Lists.newArrayList();
                List<String> usernameList = link.getUsernameList();
                String username = link.getUsername();
                if (CollectionUtils.isNotEmpty(usernameList)) {
                    wechatIds.addAll(usernameList);
                } else if (StringUtils.isNotBlank(username)) {
                    wechatIds.add(username);
                }
                if (CollectionUtils.isNotEmpty(wechatIds)) {
                    map.put("usernameList", wechatIds);
                }
                String qrcode = link.getQrcode();
                if (StringUtils.isNotBlank(qrcode)) {
                    map.put("qrcode", qrcode);
                }
                return;
            }
        }
    }

    private static Map<String, String> getReplacedContent(List<JoinGroupMsg.Link> linklist) {
        if (CollectionUtils.isEmpty(linklist)) {
            return null;
        }
        Map<String, String> map = Maps.newHashMap();
        linklist.forEach(link -> {
            String name = link.getName();
            String value = "";
            List<JoinGroupMsg.Member> memberList = link.getMemberList();
            String plain = link.getPlain();
            String title = link.getTitle();
            if (CollectionUtils.isNotEmpty(memberList)) {
                String separator = link.getSeparator();
                if (Objects.nonNull(separator)) {
                    value = buildMemberListStr(memberList, separator);
                } else {
                    value = memberList.get(0).getNickname();
                }
            } else if (Objects.nonNull(title)) {
                value = title;

            } else if (Objects.nonNull(plain)) {
                value = plain;
            }
            map.put(name, value);
        });
        return map;
    }

    private static String buildMemberListStr(List<JoinGroupMsg.Member> memberList, String separator) {
        StringBuilder builder = new StringBuilder();
        int size = memberList.size();
        for (int i = 0; i < size; i++) {
            builder.append(memberList.get(i).getNickname());
            if (i != size - 1) {
                builder.append(separator);
            }
        }
        return builder.toString();
    }

    private static String getInvalidMsg(String msg) throws Exception {
        String start = "<sysmsg";
        String end = "</sysmsg>";
        int index = msg.indexOf(start);
        int endIndex = msg.indexOf(end);
        if (index == -1 || endIndex == -1) {
            throw new Exception("消息内容格式错误，无法解析");
        }
        msg = msg.substring(index, endIndex + end.length());
        //msg = msg.replaceAll("\\\\\"", "\"");
        return msg;
    }

}
