/**
 * Kuaikeguanjia.com Inc. Copyright (c) 2014-2020 All Rights Reserved.
 */
package com.kuaike.skynet.rc.service.riskControl.enums;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import java.util.List;
import java.util.Map;

/**
 * @title RcCheckType
 * @author yanmaoyuan
 * @date 2020年6月16日
 * @version 1.0
 */
public enum RcCheckType {

    // 群消息检测
    TEXT("text", "发广告", "发了敏感词"),

    BCARD("bcard", "发广告", "发名片"),

    VIDEO("video", "发广告", "发小视频"),

    LINK("link", "发广告", "发链接分享"),

    MINI_PROGRAM("miniProgram", "发广告", "发小程序"),

    // 防骚扰
    RATE_LIMIT("rateLimit", "刷屏", "连续%d秒发送了%d条消息"),

    LINE_LIMIT("lineLimit", "刷屏", "信息大于%d行"),

    CHAR_LIMIT("charLimit", "刷屏", "信息长度超过%d字"),

    // 群成员昵称检测
    NICKNAME("nickname", "昵称包含敏感词", "昵称包含敏感词"),

    // 私加小号
    ADD_ROBOT("addRobot", "私加小号", "私加小号"),

    JOIN_BY_QRCODE("joinByQrcode", "扫码入群", "非白名单用户提供二维码入群"),

    JOIN_BY_INVITE("joinByInvite", "邀请入群", "非白名单用户邀请他人入群"),
    ;

    private final static Map<String, RcCheckType> CACHE;

    /**
     * 防广告操作中，所需检测的消息类型
     */
    private final static List<RcCheckType> MESSAGE_TYPES = Lists.newArrayList(//
        TEXT, // 文本
        BCARD, // 个人名片
        LINK, // 链接卡片
        VIDEO, // 小视频
        MINI_PROGRAM// 小程序
    );

    /**
     * 防骚扰操作中，所需检测的行为类型
     */
    private final static List<RcCheckType> ANTI_HARNESS_TYPES = Lists.newArrayList(//
        RATE_LIMIT,// 消息发送速率
        LINE_LIMIT,// 消息最大行数
        CHAR_LIMIT// 消息最大字符数
    );
    
    /**
     * 防入群操作中，所需检测的行为类型
     */
    private final static List<RcCheckType> ANTI_JOIN_GROUP_TYPES = Lists.newArrayList(//
        JOIN_BY_QRCODE,// 扫描二维码入群
        JOIN_BY_INVITE // 被邀请入群
        );

    static {
        CACHE = Maps.newHashMap();
        for (RcCheckType t : values()) {
            CACHE.put(t.name, t);
        }
    }

    private String name;

    private String desc;

    private String reason;
    
    private RcCheckType(String name, String desc, String reason) {
        this.name = name;
        this.desc = desc;
        this.reason = reason;
    }

    /**
     * @return the name
     */
    public String getName() {
        return name;
    }

    /**
     * @return the desc
     */
    public String getDesc() {
        return desc;
    }

    public String getReason() {
        return reason;
    }

    /**
     * @param type
     * @return 
     */
    public static RcCheckType getType(String type) {
        return CACHE.get(type);
    }

    public static List<RcCheckType> getMessageTypes() {
        return Lists.newArrayList(MESSAGE_TYPES);
    }
    
    public static List<RcCheckType> getAntiHarnessTypes() {
        return Lists.newArrayList(ANTI_HARNESS_TYPES);
    }
    
    public static List<RcCheckType> getAntiJoinGroupTypes() {
        return Lists.newArrayList(ANTI_JOIN_GROUP_TYPES);
    }
}