/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.entity.message.resp;

import cn.kinyun.scrm.weixin.sdk.enums.WxMsgType;
import cn.kinyun.scrm.weixin.sdk.utils.MsgXmlBuilder;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * 将消息转发到客服
 * 
 * <p>
 * 如果公众号处于开发模式，普通微信用户向公众号发消息时，微信服务器会先将消息POST到开发者填写的url上，如果希望将消息转发到客服系统，则需要开发者在响应包中返回MsgType为transfer_customer_service的消息，微信服务器收到响应后会把当次发送的消息转发至客服系统。
 * </p>
 * 
 * <p>
 * 您也可以在返回transfer_customer_service消息时，在XML中附上TransInfo信息指定分配给某个客服帐号。
 * </p>
 * 
 * <p>
 * 这里特别要注意，只针对微信用户发来的消息才进行转发，而对于其他任何事件（比如菜单点击、地理位置上报等）都不应该转接，否则客服在客服系统上就会看到一些无意义的消息了。
 * </p>
 * 
 * @title TransferCustomerService
 * @desc 将消息转发到客服
 * @author yanmaoyuan
 * @date 2019年4月30日
 * @version 1.0
 * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1458557405">将消息转发到客服</a>
 */
@Data
@EqualsAndHashCode(callSuper = true)
@ToString(callSuper = true)
public class TransferCustomerService extends BaseRespMsg {

    private static final long serialVersionUID = 5263324280655578067L;

    /**
     * 消息转发到指定客服
     * 
     * <p>
     * 如果您有多个客服人员同时登录了客服并且开启了自动接入在进行接待，每一个客户的消息转发给客服时，多客服系统会将客户分配给其中一个客服人员。如果您希望将某个客户的消息转给指定的客服来接待，可以在返回transfer_customer_service消息时附上TransInfo信息指定一个客服帐号。
     * </p>
     * <p>
     * 需要注意，如果指定的客服没有接入能力(不在线、没有开启自动接入或者自动接入已满)，该用户会被直接接入到指定客服，不再通知其它客服，不会被其他客服接待。建议在指定客服时，先查询客服的接入能力（获取在线客服接待信息接口），指定到有能力接入的客服，保证客户能够及时得到服务。
     * </p>
     */
    private String kfAccount;

    public TransferCustomerService() {
        this.msgType = WxMsgType.TransferCustomerService.getValue();
    }

    @Override
    public String toXml() {

        MsgXmlBuilder builder = new MsgXmlBuilder(super.toXml());

        // 指定客服
        if (kfAccount != null) {
            builder.append("<TransInfo>");
            builder.addCData("KfAccount", kfAccount);
            builder.append("</TransInfo>");
        }
        builder.surroundWith("xml");

        return builder.toString();
    }
}