/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api.analysis;

import cn.kinyun.scrm.weixin.sdk.entity.analysis.AnalysisReq;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.api.ApiSummary;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.api.ApiSummaryHour;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 微信接口数据分析接口
 * 
 * <p>在公众号登录授权机制的权限集划分中，接口分析数据接口属于帐号服务权限。</p>
 * 
 * @see <a href="https://developers.weixin.qq.com/doc/offiaccount/Analytics/Analytics_API.html">数据统计-接口分析</a>
 * @title WxApiAnalysisAPI
 * @author yanmaoyuan
 * @date 2019年9月24日
 * @version 1.0
 */
@Slf4j
@Component
public class WxApiAnalysisAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 获取接口分析数据 POST
     */
    @Value("${wx.analysis.getinterfacesummary}")
    private String wxAnalysisGetInterfaceSummary;

    /**
     * 获取接口分析分时数据 POST
     */
    @Value("${wx.analysis.getinterfacesummaryhour}")
    private String wxAnalysisGetInterfaceSummaryHour;

    /**
     * 获取接口分析数据
     * 
     * <p>
     * 最大时间跨度:30天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public ApiSummary getApiSummary(@NonNull String accessToken, @NonNull AnalysisReq params) throws WeixinException {
        log.info("get interface summary with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetInterfaceSummary, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<ApiSummary> response = restTemplate.postForEntity(url, request, ApiSummary.class);

        ApiSummary result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取接口分析分时数据
     * 
     * <p>
     * 最大时间跨度:1天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public ApiSummaryHour getApiSummaryHour(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get interface summary hour with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetInterfaceSummaryHour, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<ApiSummaryHour> response = restTemplate.postForEntity(url, request, ApiSummaryHour.class);

        ApiSummaryHour result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}