/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.message.mass.resp.MsgId;
import cn.kinyun.scrm.weixin.sdk.entity.template.*;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import cn.kinyun.scrm.weixin.sdk.utils.JsonUtil;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.List;
import java.util.Map;

/**
 * 微信模板消息接口
 * 
 * <p>
 * 模板消息仅用于公众号向用户发送重要的服务通知，只能用于符合其要求的服务场景中，如信用卡刷卡通知，商品购买成功通知等。不支持广告等营销类消息以及其它所有可能对用户造成骚扰的消息。
 * </p>
 * 
 * @title WxTemplateMsgAPI
 * @desc 微信模板消息接口
 * @author yanmaoyuan
 * @date 2019年4月28日
 * @version 1.0
 * @see <a href="https://mp.weixin.qq.com/wiki?t=resource/res_main&id=mp1433751277">模板消息接口</a>
 */
@Slf4j
@Component
public class WxTemplateMsgAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 设置所属行业 POST
     */
    @Value("${wx.template.industry.set}")
    private String wxTemplateIndustrySet;

    /**
     * 获取设置的行业信息 GET
     */
    @Value("${wx.template.industry.get}")
    private String wxTemplateIndustryGet;

    /**
     * 获得模板ID POST
     */
    @Value("${wx.template.add}")
    private String wxTemplateAdd;

    /**
     * 获取模板列表 GET
     */
    @Value("${wx.template.getAll}")
    private String wxTemplateGetAll;

    /**
     * 删除模板 POST
     */
    @Value("${wx.template.delete}")
    private String wxTemplateDelete;

    /**
     * 发送模板消息 POST
     */
    @Value("${wx.message.template.send}")
    private String wxMessageTemplateSend;

    /**
     * 设置所属行业
     * 
     * <p>
     * 设置行业可在微信公众平台后台完成，每月可修改行业1次，帐号仅可使用所属行业中相关的模板，为方便第三方开发者，提供通过接口调用的方式来修改账号所属行业。
     * <p>
     * 
     * @param accessToken 接口调用凭证
     * @param params 公众号模板消息所属行业编号
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public void setIndustry(@NonNull String accessToken, @NonNull SetIndustry params) throws WeixinException {
        log.info("set industry with params={}", params);
        Preconditions.checkArgument(StringUtils.isNoneBlank(params.getIndustryId1()), "主营行业编号为空");
        Preconditions.checkArgument(StringUtils.isNoneBlank(params.getIndustryId2()), "副营行业编号为空");

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        HttpEntity<SetIndustry> request = new HttpEntity<SetIndustry>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxTemplateIndustrySet, accessToken);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        WeixinException.isSuccess(response.getBody());// 处理错误码
    }

    /**
     * 获取设置的行业信息
     * 
     * <p>
     * 获取帐号设置的行业信息。可登录微信公众平台，在公众号后台中查看行业信息。为方便第三方开发者，提供通过接口调用的方式来获取帐号所设置的行业信息。
     * </p>
     * 
     * @param accessToken 接口调用凭证
     * @return 帐号设置的行业信息
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public IndustryInfo getIndustry(@NonNull String accessToken) throws WeixinException {
        log.info("get industry");

        // 发送请求
        String url = MessageFormat.format(wxTemplateIndustryGet, accessToken);
        ResponseEntity<IndustryInfo> response = restTemplate.getForEntity(url, IndustryInfo.class);

        IndustryInfo result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码

        return result;
    }

    /**
     * 添加私有模板，获得模板ID
     * 
     * <p>
     * 从行业模板库选择模板到帐号后台，获得模板ID的过程可在微信公众平台后台完成。为方便第三方开发者，提供通过接口调用的方式来获取模板ID。
     * </p>
     * 
     * @param accessToken 接口调用凭证
     * @param addTemplate 模板库中模板的编号，有“TM**”和“OPENTMTM**”等形式
     * @return 模板ID
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public String addPrivateTemplate(@NonNull String accessToken, @NonNull AddTemplate addTemplate)
        throws WeixinException {
        log.info("add template with shortId={}", addTemplate.getTemplateIdShort());
        Preconditions.checkArgument(StringUtils.isNoneBlank(addTemplate.getTemplateIdShort()), "模板短ID为空");
        Preconditions.checkArgument(addTemplate.getKeywordNameList() != null && !addTemplate.getKeywordNameList().isEmpty(), "公众号模板消息所属行业编号为空");

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        Map<String, Object> params = Maps.newHashMap();
        params.put("template_id_short", addTemplate.getTemplateIdShort());
        params.put("keyword_name_list", addTemplate.getKeywordNameList());
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxTemplateAdd, accessToken);
        ResponseEntity<PrivateTemplate> response = restTemplate.postForEntity(url, request, PrivateTemplate.class);

        PrivateTemplate result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码

        return result.getTemplateId();
    }

    /**
     * 获取模板列表
     * 
     * <p>
     * 获取已添加至帐号下所有模板列表，可在微信公众平台后台中查看模板列表信息。为方便第三方开发者，提供通过接口调用的方式来获取帐号下所有模板信息。
     * </p>
     * 
     * @param accessToken 接口调用凭证
     * @return 模板列表
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public List<PrivateTemplate> getAllPrivateTemplate(@NonNull String accessToken) throws WeixinException {
        log.info("get all private template");

        // 发送请求
        String url = MessageFormat.format(wxTemplateGetAll, accessToken);
        ResponseEntity<PrivateTemplateList> response = restTemplate.getForEntity(url, PrivateTemplateList.class);

        PrivateTemplateList result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码

        return result.getList();
    }

    /**
     * 删除模板
     * 
     * @param accessToken 接口调用凭证
     * @param templateId 模板ID
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public void deletePrivateTemplate(@NonNull String accessToken, @NonNull String templateId) throws WeixinException {
        log.info("delete private template with templateId={}", templateId);
        Preconditions.checkArgument(StringUtils.isNoneBlank(templateId), "模板ID为空");

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        Map<String, Object> params = Maps.newHashMap();
        params.put("template_id", templateId);
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        // 发送请求
        String url = MessageFormat.format(wxTemplateDelete, accessToken);
        ResponseEntity<ErrorCode> response = restTemplate.postForEntity(url, request, ErrorCode.class);

        WeixinException.isSuccess(response.getBody());// 处理错误码
    }

    /**
     * 发送模板消息
     * 
     * @param accessToken 接口调用凭证
     * @param params 模板消息
     * @return 消息ID
     * @throws WeixinException 错误时微信会返回错误码等信息，请根据错误码查询错误信息
     */
    public MsgId sendTemplateMsg(@NonNull String accessToken, @NonNull TemplateMsg params) throws WeixinException {
        log.info("send template msg with params={}", params);
        Preconditions.checkArgument(StringUtils.isNoneBlank(params.getToUserName()), "接受者的openid为空");
        Preconditions.checkArgument(StringUtils.isNoneBlank(params.getTemplateId()), "消息模板ID为空");
        Preconditions.checkArgument(params.getData() != null, "模板参数为空");
        Preconditions.checkArgument(!params.getData().isEmpty(), "模板参数为空");
        Preconditions.checkArgument(params.getMiniProgram() == null || // 小程序是可选参数
            StringUtils.isNoneBlank(params.getMiniProgram().getAppId()), "小程序appid为空");// 如果设置了小程序，则appid为必填

        // 构造请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        // 这里直接使用json序列化后getBytes()
        // Jackson在序列化字符串时，先把字符串转换成char[]再逐一序列化。
        // 一个emoji占两个char，会被转化成\\uXXXX\\uXXXX这种形式，在公众号就看不到emoji了。
        byte[] data = JsonUtil.obj2Str(params).getBytes();
        HttpEntity<?> request = new HttpEntity<>(data, headers);

        // 发送请求
        String url = MessageFormat.format(wxMessageTemplateSend, accessToken);
        ResponseEntity<MsgId> response = restTemplate.postForEntity(url, request, MsgId.class);

        MsgId msgId = response.getBody();
        WeixinException.isSuccess(msgId);// 处理错误码

        return msgId;
    }

}