/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.entity.message.resp;

import cn.kinyun.scrm.weixin.sdk.enums.WxMsgType;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.google.common.collect.Lists;
import cn.kinyun.scrm.weixin.sdk.utils.MsgXmlBuilder;

import java.util.List;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.ToString;

/**
 * 回复图文消息
 * 
 * @title NewsMsg
 * @desc 回复图文消息
 * @author yanmaoyuan
 * @date 2019年4月25日
 * @version 1.0
 */
@Data
@EqualsAndHashCode(callSuper = true)
@ToString(callSuper = true)
public class NewsMsg extends BaseRespMsg {

    private static final long serialVersionUID = -8211099268651467294L;

    /**
     * 图文消息个数的最大值。
     * 
     * 当用户发送文本、图片、视频、图文、地理位置这五种消息时，开发者只能回复1条图文消息；
     * 
     * 其余场景最多可回复8条图文消息
     */
    private static final int MAX_ARTICLE_COUNT = 8;

    /**
     * 图新消息的最大个数，初始化为默认值
     */
    @JsonIgnore
    private int maxSize = MAX_ARTICLE_COUNT;

    /**
     * 图文消息信息，注意，如果图文数超过限制，则将只发限制内的条数
     */
    private List<Article> articles;

    public NewsMsg() {
        this.msgType = WxMsgType.News.getValue();
        this.maxSize = MAX_ARTICLE_COUNT;
        this.articles = Lists.newArrayListWithCapacity(MAX_ARTICLE_COUNT);
    }

    public NewsMsg(int maxSize) {
        this.msgType = WxMsgType.News.getValue();
        this.setMaxSize(maxSize);
        this.articles = Lists.newArrayListWithCapacity(this.maxSize);
    }

    public NewsMsg(List<Article> articles) {
        this.msgType = WxMsgType.News.getValue();
        this.maxSize = MAX_ARTICLE_COUNT;
        this.articles = Lists.newArrayListWithCapacity(MAX_ARTICLE_COUNT);
        this.setArticles(articles);
    }

    /**
     * 设置图文消息的最大数量
     * 
     * @param maxSize
     */
    public void setMaxSize(int maxSize) {
        if (maxSize < MAX_ARTICLE_COUNT && maxSize >= 1) {
            this.maxSize = maxSize;
        }
    }

    /**
     * 设置图文消息
     * 
     * @param articles 图文消息
     */
    public void setArticles(List<Article> articles) {
        if (this.articles.size() > 0) {
            this.articles.clear();
        }
        this.articles.addAll(articles);
    }

    /**
     * 添加一条图文消息
     * 
     * @param title 标题
     */
    public void add(String title) {
        add(title, null, null, null);
    }

    /**
     * 添加一条图文消息
     * 
     * @param title 标题
     * @param url 正文URL
     */
    public void add(String title, String url) {
        add(title, null, null, url);
    }

    /**
     * 添加一条图文消息
     * 
     * @param title 标题
     * @param picUrl 图片URL
     * @param url 正文URL
     */
    public void add(String title, String picUrl, String url) {
        add(new Article(title, null, picUrl, url));
    }

    /**
     * 添加一条图文消息
     * 
     * @param title 标题
     * @param description 描述
     * @param picUrl 图片URL
     * @param url 正文URL
     */
    public void add(String title, String description, String picUrl, String url) {
        add(new Article(title, description, picUrl, url));
    }

    /**
     * 添加一条图文消息
     * 
     * @param article 图文消息
     */
    public void add(Article article) {
        if (article != null && articles.size() < maxSize) {
            articles.add(article);
        }
    }

    @Override
    public String toXml() {
        MsgXmlBuilder builder = new MsgXmlBuilder(super.toXml());

        builder.addTag("ArticleCount", String.valueOf(articles.size()));

        builder.append("<Articles>");
        for (int i = 0; i < maxSize && i < articles.size(); i++) {
            Article article = articles.get(i);
            MsgXmlBuilder subBuilder = new MsgXmlBuilder();

            subBuilder.addCData("Title", article.getTitle());
            subBuilder.addCData("Description", article.getDescription());
            subBuilder.addCData("PicUrl", article.getPicUrl());
            subBuilder.addCData("Url", article.getUrl());
            subBuilder.surroundWith("item");

            builder.append(subBuilder.toString());
        }
        builder.append("</Articles>");

        return builder.toString();
    }
}