package cn.kinyun.scrm.weixin.sdk.api.applet;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.req.Add2TemplateReq;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.req.TemplateIdReq;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.TemplateDraftListResp;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.TemplateListResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Objects;

/**
 * @author zhangjing
 * @desc https://developers.weixin.qq.com/doc/oplatform/Third-party_Platforms/2.0/api/ThirdParty/code_template/gettemplatedraftlist.html
 * @date 2022/1/17 4:34 下午
 */
@Slf4j
@Component
public class AppletTemplateAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 小程序模板管理-获取代码草稿列表 GET
     */
    @Value("${wx.applet.template.manager.draft_list}")
    private String appletTemplateDraftListUrl;

    /**
     * 小程序模板管理-将草稿添加到代码模板库  POST
     */
    @Value("${wx.applet.template.manager.add2Template}")
    private String appletTemplateAdd2TemplateUrl;

    /**
     * 小程序模板管理-获取代码模板列表  GET
     */
    @Value("${wx.applet.template.manager.get}")
    private String appletTemplateGetUrl;

    /**
     * 小程序模板管理-删除指定代码模板  POST
     */
    @Value("${wx.applet.template.manager.delete}")
    private String appletTemplateDeleteUrl;

    /**
     * 获取代码草稿列表
     *
     * @param accessToken
     * @return
     */
    public TemplateDraftListResp getDraftList(@NonNull String accessToken) {
        log.info("getDraftList");

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");

        ResponseEntity<TemplateDraftListResp> resp = null;

        // 发送请求
        String url = MessageFormat.format(appletTemplateDraftListUrl, accessToken);
        resp = restTemplate.getForEntity(url, TemplateDraftListResp.class);

        TemplateDraftListResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 将草稿添加到代码模板库
     *
     * @param accessToken
     * @return
     */
    public ErrorCode add2Template(@NonNull String accessToken, @NonNull Add2TemplateReq req) {
        log.info("add2Template with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<Add2TemplateReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(appletTemplateAdd2TemplateUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取代码模板列表
     *
     * @param accessToken
     * @return
     * @poram templateType 可选是0（对应普通模板）和1（对应标准模板），如果不填，则返回全部的
     */
    public TemplateListResp getTemplate(@NonNull String accessToken,Integer templateType) {
        log.info("getTemplate with templateType={}", templateType);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");

        ResponseEntity<TemplateListResp> resp = null;

        // 发送请求
        String url = MessageFormat.format(appletTemplateGetUrl, accessToken);
        resp = restTemplate.getForEntity(url, TemplateListResp.class, templateType);

        TemplateListResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 删除指定代码模板
     *
     * @param accessToken
     * @return
     */
    public ErrorCode delTemplate(@NonNull String accessToken, Integer templateType) {
        log.info("delTemplate with templateType={}", templateType);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(templateType != null, "templateType不能为空");

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        TemplateIdReq req = new TemplateIdReq();
        req.setTemplateId(templateType);
        HttpEntity<TemplateIdReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(appletTemplateDeleteUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
