/*
 * 微信公众平台(JAVA) SDK
 *
 * Copyright (c) 2014, Ansitech Network Technology Co.,Ltd All rights reserved.
 * 
 * http://www.weixin4j.org/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.kinyun.scrm.weixin.sdk.utils;

import cn.kinyun.scrm.weixin.sdk.entity.media.Attachment;
import com.alibaba.fastjson.JSONObject;

import org.apache.commons.io.IOUtils;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;

/**
 * HttpClient业务
 *
 * @author yangqisheng
 * @since 0.0.1
 */
public class HttpClient {

    private static final int CONNECTION_TIMEOUT = 30000;
    private static final int READ_TIMEOUT = 30000;
    private static final String POST = "POST";

    /**
     * 上传文件
     *
     * @param url 上传地址
     * @param file 上传文件对象
     * @return 服务器上传响应结果
     * @throws IOException IO异常
     * @throws NoSuchAlgorithmException 算法异常
     * @throws NoSuchProviderException 私钥异常
     * @throws KeyManagementException 密钥异常
     */
    public String upload(String url, File file) throws IOException,
            NoSuchAlgorithmException, NoSuchProviderException,
            KeyManagementException {
        HttpURLConnection http = null;
        StringBuffer bufferRes = new StringBuffer();
        try {
            // 定义数据分隔线 
            String BOUNDARY = "----WebKitFormBoundaryiDGnV9zdZA1eM1yL";
            //创建https请求连接
            http = getHttpURLConnection(url);
            //设置header和ssl证书
            setHttpHeader(http, POST);
            //不缓存
            http.setUseCaches(false);
            //保持连接
            http.setRequestProperty("connection", "Keep-Alive");
            //设置文档类型
            http.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + BOUNDARY);

            //定义输出流
            OutputStream out = null;
            //定义输入流
            try(DataInputStream dataInputStream = new DataInputStream(new FileInputStream(file));) {
                out = new DataOutputStream(http.getOutputStream());
                byte[] end_data = ("\r\n--" + BOUNDARY + "--\r\n").getBytes();// 定义最后数据分隔线  
                StringBuilder sb = new StringBuilder();
                sb.append("--");
                sb.append(BOUNDARY);
                sb.append("\r\n");
                sb.append("Content-Disposition: form-data;name=\"media\";filename=\"").append(file.getName()).append("\"\r\n");
                sb.append("Content-Type:application/octet-stream\r\n\r\n");
                byte[] data = sb.toString().getBytes();
                out.write(data);
                //读取文件流
                int bytes;
                byte[] bufferOut = new byte[1024];
                while ((bytes = dataInputStream.read(bufferOut)) != -1) {
                    out.write(bufferOut, 0, bytes);
                }
                out.write("\r\n".getBytes()); //多个文件时，二个文件之间加入这个  

                out.write(end_data);
                out.flush();
            } finally {
                if (out != null) {
                    out.close();
                }
            }

            // 定义BufferedReader输入流来读取URL的响应  
            InputStream ins = http.getInputStream();
            try(BufferedReader read = new BufferedReader(new InputStreamReader(ins, StandardCharsets.UTF_8))) {
                String valueString;
                bufferRes = new StringBuffer();
                while ((valueString = read.readLine()) != null) {
                    bufferRes.append(valueString);
                }
            } finally {
                if (ins != null) {
                    ins.close();
                }
            }
        } finally {
            if (http != null) {
                // 关闭连接
                http.disconnect();
            }
        }
        return bufferRes.toString();
    }

    /**
     * 下载附件
     *
     * @param url 附件地址
     * @return 附件对象
     * @throws IOException IO异常
     */
    public Attachment download(String url) throws IOException {
        Attachment attachment = new Attachment();
        URL _url = new URL(url);
        HttpURLConnection http = (HttpURLConnection) _url.openConnection();
        //设置头
        setHttpHeader(http, "GET");
        if (http.getContentType().equalsIgnoreCase("text/plain")) {
            // 定义BufferedReader输入流来读取URL的响应  
            InputStream in = http.getInputStream();
            BufferedReader read = new BufferedReader(new InputStreamReader(in, StandardCharsets.UTF_8));
            String valueString;
            StringBuilder bufferRes = new StringBuilder();
            while ((valueString = read.readLine()) != null) {
                bufferRes.append(valueString);
            }
            in.close();
            attachment.setErrCode(-1);
            attachment.setErrMsg(bufferRes.toString());
        } else if (http.getContentType().contains("application/json")) {
            // 定义BufferedReader输入流来读取URL的响应  
            InputStream in = http.getInputStream();
            BufferedReader read = new BufferedReader(new InputStreamReader(in, StandardCharsets.UTF_8));
            String valueString;
            StringBuilder bufferRes = new StringBuilder();
            while ((valueString = read.readLine()) != null) {
                bufferRes.append(valueString);
            }
            in.close();
            String jsonString = bufferRes.toString();
            JSONObject result = JSONObject.parseObject(jsonString);
            if (result.containsKey("errcode") && result.getIntValue("errcode") != 0) {
                attachment.setErrCode(result.getInteger("errcode"));
                attachment.setErrMsg(result.getString("errmsg"));
            } else {
                //未知格式
                attachment.setErrCode(-1);
                attachment.setErrMsg(jsonString);
            }
        } else {
            String ds = http.getHeaderField("Content-disposition");
            // 这里返回的是 attachment; filename=xxxxxxx.mp4
            // 也有可能是 form-data; name="aaa"; filename="xxxxxxxx.mp4"
            String fileName = "";
            if (ds.contains("attachment;")) {
                fileName = ds.substring(ds.indexOf("filename=") + 10, ds.length());
            } else if (ds.contains("form-data;")) {
                fileName = ds.substring(ds.indexOf("filename=\"") + 10, ds.length() - 1);
            }
            attachment.setFileName(fileName);
            attachment.setContentLength(http.getHeaderField("Content-Length"));
            attachment.setContentType(http.getHeaderField("Content-Type"));

            attachment.setInputStream(new ByteArrayInputStream(IOUtils.toByteArray(http.getInputStream())));
        }
        
        http.disconnect();
        return attachment;
    }

    /**
     * 获取http请求连接
     *
     * @param url 连接地址
     * @return http连接对象
     * @throws IOException IO异常
     */
    private HttpURLConnection getHttpURLConnection(String url) throws IOException {
        URL urlGet = new URL(url);
        HttpURLConnection con = (HttpURLConnection) urlGet.openConnection();
        return con;
    }

    private void setHttpHeader(HttpURLConnection httpUrlConnection, String method)
            throws IOException {
        //设置header信息
        httpUrlConnection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
        //设置User-Agent信息
        httpUrlConnection.setRequestProperty("User-Agent", "Mozilla/5.0 (Windows NT 6.3; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/33.0.1750.146 Safari/537.36");
        //设置可接受信息
        httpUrlConnection.setDoOutput(true);
        //设置可输入信息
        httpUrlConnection.setDoInput(true);
        //设置请求方式
        httpUrlConnection.setRequestMethod(method);
        //设置连接超时时间
        httpUrlConnection.setConnectTimeout(CONNECTION_TIMEOUT);
        //设置请求超时
        httpUrlConnection.setReadTimeout(READ_TIMEOUT);
        //设置编码
        httpUrlConnection.setRequestProperty("Charsert", "UTF-8");
    }
}
