/*
 * kuaike.com Inc. Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.emoji;

import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 微信Emoji转化工具
 * 
 * @title WxEmojiUtils
 * @desc 微信Emoji转化工具
 * @author yanmaoyuan
 * @date 2019年6月22日
 * @version 1.0
 */
@Slf4j
public final class WxEmojiUtils {

    private WxEmojiUtils() {}

    /**
     * 解码用正则表达式
     */
    private static final String DECODE_REGEX = "[\\uE001-\\uE5FF]";

    /**
     * 解码表
     */
    private static final Map<String, String> DECODE_MAP = new HashMap<>();

    /**
     * 编码用正则表达式
     */
    private static final String ENCODE_REGEX = "[\\uD83C\\uDDE0-\\uD83C\\uDDFF][\\uD83C\\uDDE0-\\uD83C\\uDDFF]|[\\uD83C\\uDC00-\\uD83C\\uDFFF]|[\\uD83D\\uDC00-\\uD83D\\uDFFF]";

    /**
     * 编码表
     */
    private static final Map<String, String> ENCODE_MAP = new HashMap<>();

    /**
     * 静态代码块，初始化正则表达式和编码表
     */
    static {
        // 读取配置中记录的emoji映射关系
        Properties prop = new Properties();
        try {
            InputStream in = WxEmojiUtils.class.getResourceAsStream("/emoji.properties");
            prop.load(in);
            prop.entrySet().forEach(it -> {
                String key = (String) it.getKey();
                String value = (String) it.getValue();
                DECODE_MAP.put(key, value);
                ENCODE_MAP.put(value, key);
            });
        } catch (IOException e) {
            log.error("load properties failed", e);
        }

    }

    /**
     * 替换字符串，将编码表中的词素转义成另一组词素。
     * 
     * @param source
     * @param regex
     * @param map
     * @return
     */
    public static String replace(String source, String regex, Map<String, String> map) {
        if (source == null || source.trim().length() == 0) {
            return source;
        }

        if (source.length() > 0) {
            Map<String, String> tmpMap = new HashMap<>();

            // 使用正则表达式检查匹配的文本
            Pattern emoji = Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
            Matcher m = emoji.matcher(source);
            while (m.find()) {
                String key = m.group();
                String value = map.get(key);
                if (value != null) {
                    tmpMap.put(key, value);
                }
            }

            // 逐一替换命中的符号
            if (!tmpMap.isEmpty()) {
                for (Entry<String, String> entry : tmpMap.entrySet()) {
                    String key = entry.getKey();
                    String value = entry.getValue();
                    source = source.replace(key, value);
                }
            }
        }
        return source;
    }

    /**
     * 判断字符串中是否包含正则式中的词素
     * 
     * @param source 字符串
     * @param regex 正则表达式
     * @return true 包含, false 不包含
     */
    public static boolean contains(String source, String regex) {
        boolean result = false;
        if (source != null && source.length() > 0) {
            Pattern p = Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
            Matcher m = p.matcher(source);
            if (m.find()) {
                result = true;
            }
        }
        return result;
    }

    /**
     * 前端无法显示软银的emoji，需要把他们转换成另一套unicode编码。
     * 
     * @param source
     * @return
     */
    public static String decode(String source) {
        return replace(source, DECODE_REGEX, DECODE_MAP);
    }

    /**
     * 将字符串中的emoji进行转换。
     * 
     * 微信对\uD83D\uDDE3之类的双字符emoji解析有问题，单字符的emoji可以正常显示。
     * 因此这里需要把多字符的emoji转码成单字符的软银emoji。
     * 
     * @param source
     * @return
     */
    public static String encode(String source) {
        return replace(source, ENCODE_REGEX, ENCODE_MAP);
    }

    /**
     * 检测字符串中是否包含软银的emoji
     * 
     * @param source
     * @return
     */
    public static boolean containsSoftbankEmoji(String source) {
        return contains(source, DECODE_REGEX);
    }

    /**
     * 检测字符串中是否包含unicode多字符emoji
     * 
     * @param source
     * @return
     */
    public static boolean containsUnicodeEmoji(String source) {
        return contains(source, ENCODE_REGEX);
    }
}