package cn.kinyun.scrm.weixin.sdk.api.shop;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.shop.req.*;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.*;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Map;
import java.util.Objects;

/**
 * @author zhangjing
 * @date 2022/1/5 7:03 下午
 */
@Slf4j
@Component
public class WxShopFundAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 资金管理接口-获取商户余额 POST
     */
    @Value("${wx.shop.funds.get_balance}")
    private String shopFundsGetBalanceUrl;

    /**
     * 资金管理接口-获取资金流水 POST
     */
    @Value("${wx.shop.funds.get_order_flow}")
    private String shopFundGetOrderFlowUrl;

    /**
     * 资金管理接口-获取资金流水列表 POST
     */
    @Value("${wx.shop.funds.scan_order_flow}")
    private String shopFundScanOrderFlowUrl;

    /**
     * 资金管理接口-获取提现记录 POST
     */
    @Value("${wx.shop.funds.get_withdraw_detail}")
    private String shopFundGetWithdrawDetailUrl;

    /**
     * 资金管理接口-获取提现记录列表 POST
     */
    @Value("${wx.shop.funds.scan_withdraw}")
    private String shopFundScanWithdrawUrl;

    /**
     * 资金管理接口-修改结算账户 POST
     */
    @Value("${wx.shop.funds.set_bank_account}")
    private String shopFundSetBankAccountUrl;

    /**
     * 资金管理接口-商户提现 POST
     */
    @Value("${wx.shop.funds.submit_withdraw}")
    private String shopFundSubmitWithdrawUrl;

    /**
     * 获取账户余额
     *
     * @param accessToken
     * @return
     */
    public FundGetBalanceResp getBalance(@NonNull String accessToken) {
        log.info("call getBalance");

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");

        ResponseEntity<FundGetBalanceResp> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        Map<String, Object> req = Maps.newHashMap();
        HttpEntity<Map<String, Object>> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopFundsGetBalanceUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, FundGetBalanceResp.class);

        FundGetBalanceResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取资金流水
     *
     * @param accessToken
     * @param req
     * @return
     */
    public FundGetOrderFlowResp getOrderFlow(@NonNull String accessToken, @NonNull FundGetOrderFlowReq req) {
        log.info("getOrderFlow with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<FundGetOrderFlowResp> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<FundGetOrderFlowReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopFundGetOrderFlowUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, FundGetOrderFlowResp.class);

        FundGetOrderFlowResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取资金流水列表
     *
     * @param accessToken
     * @param req
     * @return
     */
    public FundGetOrderFlowListResp getOrderFlowList(@NonNull String accessToken, @NonNull FundGetOrderFlowListReq req) {
        log.info("getOrderFlowList with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<FundGetOrderFlowListResp> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<FundGetOrderFlowListReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopFundScanOrderFlowUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, FundGetOrderFlowListResp.class);

        FundGetOrderFlowListResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取提现记录
     *
     * @param accessToken
     * @param req
     * @return
     */
    public FundGetWithdrawResp getWithdrawDetail(@NonNull String accessToken, @NonNull FundGetWithdrawDetailReq req) {
        log.info("getWithdrawDetail with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<FundGetWithdrawResp> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<FundGetWithdrawDetailReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopFundGetWithdrawDetailUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, FundGetWithdrawResp.class);

        FundGetWithdrawResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取提现记录列表
     *
     * @param accessToken
     * @param req
     * @return
     */
    public FundGetScanWithdrawResp scanWithdrawList(@NonNull String accessToken, @NonNull FundGetScanWithdrawReq req) {
        log.info("scanWithdrawList with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<FundGetScanWithdrawResp> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<FundGetScanWithdrawReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopFundScanWithdrawUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, FundGetScanWithdrawResp.class);

        FundGetScanWithdrawResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 修改结算账户
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode setBankAccount(@NonNull String accessToken, @NonNull FundSetBankAccountReq req) {
        log.info("setBankAccount with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        ResponseEntity<ErrorCode> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<FundSetBankAccountReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopFundSetBankAccountUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 商户提现
     *
     * @param accessToken
     * @param req
     * @return
     */
    public FundSubmitWithdrawResp submitWithdraw(@NonNull String accessToken, @NonNull FundSubmitWithdrawReq req) {
        log.info("submitWithdraw with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        ResponseEntity<FundSubmitWithdrawResp> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<FundSubmitWithdrawReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopFundSubmitWithdrawUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, FundSubmitWithdrawResp.class);

        FundSubmitWithdrawResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
