/**
 * kuaike.com Inc.
 * Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api.account;

import cn.kinyun.scrm.weixin.sdk.entity.account.ShorturlResult;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.collect.Maps;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Map;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 微信长链接转短链接接口
 * 
 * @see <a href="https://developers.weixin.qq.com/doc/offiaccount/Account_Management/URL_Shortener.html">长链接转短链接接口</a>
 * @title WxShorturlAPI
 * @author yanmaoyuan
 * @date 2019年11月5日
 * @version 1.0
 */
@Component
@Slf4j
public class WxShorturlAPI {

    @Autowired
    private RestTemplate restTemplate;

    @Value("${wx.shorturl}")
    private String wxShorturl;

    /**
     * 创建临时二维码
     * 
     * @throws WeixinException
     */
    public ShorturlResult shorturl(@NonNull String accessToken, @NonNull String longUrl)
        throws WeixinException {
        log.info("shorturl url={}", longUrl);

        /**
         * 构造请求头
         */
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        /**
         * 构造请求体
         */
        // 请求参数
        Map<String, Object> params = Maps.newHashMap();
        params.put("action", "long2short");
        params.put("long_url", longUrl);

        // 设置消息体
        HttpEntity<Map<String, Object>> request = new HttpEntity<Map<String, Object>>(params, headers);

        /**
         * 发送请求
         */
        String url = MessageFormat.format(wxShorturl, accessToken);
        ResponseEntity<ShorturlResult> response =
            restTemplate.postForEntity(url, request, ShorturlResult.class);

        /**
         * 处理错误码
         */
        ShorturlResult result = response.getBody();
        WeixinException.isSuccess(result);

        return result;
    }
}