/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.sns.SnsAccessToken;
import cn.kinyun.scrm.weixin.sdk.entity.sns.SnsUserinfo;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * @author yanmaoyuan
 * @version 1.0
 * @title WxOAuth2API
 * @desc 微信开放平台授权接口
 * @date 2019年4月22日
 */
@Slf4j
@Component
public class WxOAuth2API {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 获取微信授权码 GET
     */
    @Value("${wx.oauth2.get_code}")
    private String wxOAuth2GetCode;

    /**
     * 获取微信授权码 GET
     */
    @Value("${wx.component.oauth2.get_code}")
    private String wxComponentOAuth2GetCode;

    /**
     * 通过code获取access_token GET
     */
    @Value("${wx.oauth2.access_token}")
    private String wxOAuth2AccessToken;

    /**
     * 通过code获取access_token GET
     */
    @Value("${wx.component.oauth2.access_token}")
    private String wxComponentOauth2AccessToken;

    /**
     * 刷新或续期access_token使用 GET
     */
    @Value("${wx.oauth2.refresh_token}")
    private String wxOAuth2RefreshToken;

    /**
     * 检验授权凭证（access_token）是否有效 GET
     */
    @Value("${wx.oauth2.auth}")
    private String wxOAuth2Auth;

    /**
     * 获取用户个人信息（UnionID机制）GET
     */
    @Value("${wx.oauth2.user_info}")
    private String wxOAuth2Userinfo;

    @Value("${wx.connect.qrconnect}")
    private String wxConnectQrconnect;

    /**
     * 获取微信授权码
     * 
     * @see <a href="https://developers.weixin.qq.com/doc/oplatform/Third-party_Platforms/Official_Accounts/official_account_website_authorization.html">代公众号发起网页授权</a>
     *
     * <p>
     * 第三方使用网站应用授权登录前请注意已获取相应网页授权作用域（scope=snsapi_login），则可以通过在PC端打开以下链接：
     * </p>
     *
     * <p>
     * 注意，授权是在微信浏览器中完成的，这是一个异步的过程。本方法将会生成一个url，调用者应该把这个url交给用户，让其点击链接后完成授权。
     * </p>
     *
     * <p>
     * 用户允许授权后，将会重定向到redirect_uri的网址上，并且带上code和state参数
     * </p>
     * <code>redirect_uri?code=CODE&state=STATE</code>
     * <p>
     * 若用户禁止授权，则重定向后不会带上code参数，仅会带上state参数
     * </p>
     * <code>redirect_uri?state=STATE</code>
     *
     * @param appId       应用唯一标识
     * @param redirectUrl 请使用urlEncode对链接进行处理
     * @param scope       应用授权作用域，拥有多个作用域用逗号（,）分隔，网页应用目前仅填写snsapi_login。
     * @param state       用于保持请求和回调的状态，授权请求后原样带回给第三方。该参数可用于防止csrf攻击（跨站请求伪造攻击），建议第三方带上该参数，可设置为简单的随机数加session进行校验。
     * @return 生成用于请求临时授权码的url
     */
    public String getOauthPageUrl(@NonNull String appId, @NonNull String redirectUrl, @NonNull String scope,
                                  String state) {
        log.info("generate oauth page url with appId={}, scope={}, state={}", appId, scope, state);

        String wxUrl = MessageFormat.format(wxOAuth2GetCode, appId, redirectUrl, scope, state);
        return wxUrl;
    }


    /**
     * 获取微信授权码
     *
     * <p>
     * 第三方使用网站应用授权登录前请注意已获取相应网页授权作用域（scope=snsapi_login），则可以通过在PC端打开以下链接：
     * </p>
     *
     * <p>
     * 注意，授权是在微信浏览器中完成的，这是一个异步的过程。本方法将会生成一个url，调用者应该把这个url交给用户，让其点击链接后完成授权。
     * </p>
     *
     * <p>
     * 用户允许授权后，将会重定向到redirect_uri的网址上，并且带上code和state参数
     * </p>
     * <code>redirect_uri?code=CODE&state=STATE</code>
     * <p>
     * 若用户禁止授权，则重定向后不会带上code参数，仅会带上state参数
     * </p>
     * <code>redirect_uri?state=STATE</code>
     *
     * @param appId          应用唯一标识
     * @param redirectUrl    请使用urlEncode对链接进行处理
     * @param scope          应用授权作用域，拥有多个作用域用逗号（,）分隔。
     * @param state          用于保持请求和回调的状态，授权请求后原样带回给第三方。该参数可用于防止csrf攻击（跨站请求伪造攻击），建议第三方带上该参数，可设置为简单的随机数加session进行校验。
     * @param componentAppId 开放平台appid
     * @return 生成用于请求临时授权码的url
     */
    public String getComponentOauthPageUrl(@NonNull String appId, @NonNull String redirectUrl,
        @NonNull String scope, String state, @NonNull String componentAppId) {
        log.info("generate oauth page url with appId={}, scope={}, state={}, redirectUrl={}", appId, scope, state, redirectUrl);
        String wxUrl = MessageFormat.format(wxComponentOAuth2GetCode, appId, redirectUrl, scope, state, componentAppId);
        log.info("login page url{}",wxUrl);
        return wxUrl;
    }

    /**
     * 通过code获取access_token
     *
     * @param appId     应用唯一标识，在微信开放平台提交应用审核通过后获得
     * @param appSecret 应用密钥AppSecret，在微信开放平台提交应用审核通过后获得
     * @param code      临时授权码
     * @return 授权凭证
     * @throws WeixinException
     */
    public SnsAccessToken accessToken(@NonNull String appId, @NonNull String appSecret, @NonNull String code)
            throws WeixinException {
        log.info("get sns access token with code={}", code);
        // 发送请求
        String url = MessageFormat.format(wxOAuth2AccessToken, appId, appSecret, code);
        ResponseEntity<SnsAccessToken> response = restTemplate.getForEntity(url, SnsAccessToken.class);

        SnsAccessToken result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码
        return result;
    }

    /**
     * 通过code获取access_token
     *
     * @param appId 应用唯一标识，在微信开放平台提交应用审核通过后获得
     * @param code  临时授权码
     * @param code  componentAppId
     * @param code  componentToken
     * @return 授权凭证
     * @throws WeixinException
     */
    public SnsAccessToken getComponentAccessToken(@NonNull String appId, @NonNull String code, @NonNull String componentAppId, @NonNull String componentToken)
            throws WeixinException {
        log.info("get sns access token with code={}", code);
        // 发送请求
        String url = MessageFormat.format(wxComponentOauth2AccessToken, appId, code, componentAppId, componentToken);
        ResponseEntity<SnsAccessToken> response = restTemplate.getForEntity(url, SnsAccessToken.class);

        SnsAccessToken result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码
        return result;
    }

    /**
     * <p>
     * 刷新或续期access_token使用
     * </p>
     *
     * <p>
     * access_token是调用授权关系接口的调用凭证，由于access_token有效期（目前为2个小时）较短，当access_token超时后，可以使用refresh_token进行刷新，access_token刷新结果有两种：
     * </p>
     * <ol>
     * <li>若access_token已超时，那么进行refresh_token会获取一个新的access_token，新的超时时间；</li>
     * <li>若access_token未超时，那么进行refresh_token不会改变access_token，但超时时间会刷新，相当于续期access_token。</li>
     * </ol>
     *
     * <p>
     * refresh_token拥有较长的有效期（30天），当refresh_token失效的后，需要用户重新授权，所以，请开发者在refresh_token即将过期时（如第29天时），进行定时的自动刷新并保存好它。
     * </p>
     *
     * @param appId        应用唯一标识
     * @param refreshToken 填写通过access_token获取到的refresh_token参数
     * @return SnsAccessToken
     * @throws WeixinException
     */
    public SnsAccessToken refressToken(@NonNull String appId, @NonNull String refreshToken) throws WeixinException {
        log.info("refresh token.");
        // 发送请求
        String url = MessageFormat.format(wxOAuth2RefreshToken, appId, refreshToken);
        ResponseEntity<SnsAccessToken> response = restTemplate.getForEntity(url, SnsAccessToken.class);

        SnsAccessToken result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码
        return result;
    }

    /**
     * 检验授权凭证（access_token）是否有效
     *
     * @param accessToken 调用接口凭证
     * @param openid      普通用户标识，对该公众帐号唯一
     * @return ErrorCode。正确返回: <code>{"errcode":0,"errmsg":"ok"}</code>
     * @throws WeixinException
     */
    public boolean auth(@NonNull String accessToken, @NonNull String openId) throws WeixinException {
        log.info("auth with openId={}", openId);

        // 发起请求
        String url = MessageFormat.format(wxOAuth2Auth, accessToken, openId);
        ResponseEntity<ErrorCode> response = restTemplate.getForEntity(url, ErrorCode.class);

        return WeixinException.isSuccess(response.getBody());
    }

    /**
     * 通过网页授权 access_token 获取用户基本信息（需授权作用域为 snsapi_userinfo）
     *
     * @param accessToken 接口调用凭证
     * @param openId      用户的openid
     * @param lang 国家地区语言版本
     * @return 用户信息
     * @throws WeixinException
     */
    public SnsUserinfo getUserinfo(@NonNull String accessToken, @NonNull String openId, @NonNull String lang) throws WeixinException {
        log.info("get sns userinfo with openId={}", openId);

        // 发起请求
        String url = MessageFormat.format(wxOAuth2Userinfo, accessToken, openId, lang);
        ResponseEntity<SnsUserinfo> response = restTemplate.getForEntity(url, SnsUserinfo.class);

        SnsUserinfo result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码
        return result;
    }

    /**
     * 获取网站应用授权登录链接
     * @param appId
     * @param redirectUrl
     * @param scope
     * @param state
     * @return
     */
    public String getQrConnectUrl(@NonNull String appId, @NonNull String redirectUrl, @NonNull String scope,
                                  String state) {
        log.info("getQrConnectUrl with appId={}, scope={}, state={}", appId, scope, state);

        return MessageFormat.format(wxConnectQrconnect, appId, redirectUrl, scope, state);
    }
}