/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api;

import cn.kinyun.scrm.weixin.sdk.entity.basic.AccessToken;
import cn.kinyun.scrm.weixin.sdk.entity.basic.Ticket;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 微信基础支持接口
 * 
 * @title WxBasicAPI
 * @desc 微信基础支持接口
 * @author yanmaoyuan
 * @date 2019年4月30日
 * @version 1.0
 */
@Slf4j
@Component
public class WxBasicAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 获取公众号接口调用凭证 GET
     */
    @Value("${wx.token.get}")
    private String wxTokenGet;

    /**
     * 获得接口临时票据 GET
     */
    @Value("${wx.ticket.getticket}")
    private String wxTicketGet;

    /**
     * 获取公众号接口调用凭证
     * 
     * @param appId 公众号appid
     * @param appSecret 公众号secret
     * @return 接口调用凭证
     * @throws WeixinException 错误时微信会返回错误码等信息
     */
    public AccessToken getAccessToken(@NonNull String appId, @NonNull String appSecret) throws WeixinException {
        log.info("get access_token for appid={}", appId);
        // 发起请求
        String url = MessageFormat.format(wxTokenGet, appId, appSecret);
        ResponseEntity<AccessToken> response = restTemplate.getForEntity(url, AccessToken.class);

        AccessToken result = response.getBody();
        WeixinException.isSuccess(result);// 处理错误码
        return result;
    }

    /**
     * 获得接口临时票据
     * 
     * @param accessToken 接口调用凭证
     * @param type 临时票据类型 (jsapi, wxcard)
     * @return 临时票据
     * @throws WeixinException
     */
    public Ticket getTicket(@NonNull String accessToken, @NonNull String type) throws WeixinException {
        log.info("get ticket with type={}", type);
        // 发起请求
        String url = MessageFormat.format(wxTicketGet, accessToken, type);
        ResponseEntity<Ticket> response = restTemplate.getForEntity(url, Ticket.class);

        Ticket ticket = response.getBody();
        WeixinException.isSuccess(ticket);// 处理错误码
        return ticket;
    }
}