/*
 * kinyun.cn Inc. Copyright (c) 2014-2021 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.emoji;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 微信表情转化工具
 * 
 * @title WxExpressionUtils
 * @desc 微信表情转化工具
 * @author yanmaoyuan
 * @date 2019年6月22日
 * @version 1.0
 */
public final class WxExpressionUtils {

    private WxExpressionUtils() {
        // DO NOT instance
    }

    private static final String[] CODE = { // 表情符号
        // QQ表情
        "/::)", "/::~", "/::B", "/::|", "/:8-)", "/::<", "/::$", "/::X", "/::Z", "/::'(", // 10
        "/::-|", "/::@", "/::P", "/::D", "/::O", "/::(", "/::+", "/:--b", "/::Q", "/::T", // 20
        "/:,@P", "/:,@-D", "/::d", "/:,@o", "/::g", "/:|-)", "/::!", "/::L", "/::>", "/::,@", // 30
        "/:,@f", "/::-S", "/:?", "/:,@x", "/:,@@", "/::8", "/:,@!", "/:!!!", "/:xx", "/:bye", // 40
        "/:wipe", "/:dig", "/:handclap", "/:&-(", "/:B-)", "/:<@", "/:@>", "/::-O", "/:>-|", "/:P-(", // 50
        "/::'|", "/:X-)", "/::*", "/:@x", "/:8*", "/:pd", "/:<W>", "/:beer", "/:basketb", "/:oo", // 60
        "/:coffee", "/:eat", "/:pig", "/:rose", "/:fade", "/:showlove", "/:heart", "/:break", "/:cake", "/:li", // 70
        "/:bome", "/:kn", "/:footb", "/:ladybug", "/:shit", "/:moon", "/:sun", "/:gift", "/:hug", "/:strong", // 80
        "/:weak", "/:share", "/:v", "/:@)", "/:jj", "/:@@", "/:bad", "/:lvu", "/:no", "/:ok", // 90
        "/:love", "/:<L>", "/:jump", "/:shake", "/:<O>", "/:circle", "/:kotow", "/:turn", "/:skip", "/:oY", // 100
        "/:#-0", "/:hiphot", "/:kiss", "/:<&", "/:&>", // 105
        // 微信表情
        "[Hey]", "[Facepalm]", "[Smirk]", "[Smart]", "[Concerned]", "[Yeah!]", "[發]", "[Packet]", "[Blessing]", "[Tea]", // 10
        "[Candle]", "[Pup]", // 12
    };

    private static final String[] WORD = { // 表情的文字描述
        // QQ表情
        "微笑", "撇嘴", "色", "发呆", "得意", "流泪", "害羞", "闭嘴", "睡", "大哭", // 10
        "尴尬", "发怒", "调皮", "呲牙", "惊讶", "难过", "酷", "冷汗", "抓狂", "吐", // 20
        "偷笑", "可爱", "白眼", "傲慢", "饥饿", "困", "惊恐", "流汗", "憨笑", "大兵", // 30
        "奋斗", "咒骂", "疑问", "嘘", "晕", "折磨", "衰", "骷髅", "敲打", "再见", // 40
        "擦汗", "抠鼻", "鼓掌", "糗大了", "坏笑", "左哼哼", "右哼哼", "哈欠", "鄙视", "委屈", // 50
        "快哭了", "阴险", "亲亲", "吓", "可怜", "菜刀", "西瓜", "啤酒", "篮球", "乒乓", // 60
        "咖啡", "饭", "猪头", "玫瑰", "凋谢", "示爱", "爱心", "心碎", "蛋糕", "闪电", // 70
        "炸弹", "刀", "足球", "瓢虫", "便便", "月亮", "太阳", "礼物", "拥抱", "强", // 80
        "弱", "握手", "胜利", "抱拳", "勾引", "拳头", "差劲", "爱你", "NO", "OK", // 90
        "爱情", "飞吻", "跳跳", "发抖", "怄火", "转圈", "磕头", "回头", "跳绳", "挥手", // 100
        "激动", "街舞", "献吻", "左太极", "右太极", // 105
        // 微信表情
        "嘿哈", "捂脸", "奸笑", "机智", "皱眉", "耶", "發", "红包", "福", "茶", // 10
        "蜡烛", "小狗", // 12
    };

    private static final int COUNT;

    /**
     * 解码用正则表达式
     */
    private static final String DECODE_REGEX;

    /**
     * 解码表
     */
    private static final Map<String, String> DECODE_MAP = new HashMap<>();//NOSONAR

    /**
     * 编码用正则表达式
     */
    private static final String ENCODE_REGEX;

    /**
     * 编码表
     */
    private static final Map<String, String> ENCODE_MAP = new HashMap<>();//NOSONAR

    /*
     * 静态代码块，初始化正则表达式和编码表
     */
    static {
        COUNT = CODE.length;

        // 初始化编码表
        for (int i = 0; i < COUNT; i++) {
            WORD[i] = "[" + WORD[i] + "]";
            DECODE_MAP.put(CODE[i], WORD[i]);
            ENCODE_MAP.put(WORD[i], CODE[i]);
        }

        // 初始化正则表达式
        DECODE_REGEX = generateRegex(CODE);

        ENCODE_REGEX = generateRegex(WORD);
    }

    /**
     * 将数组中的词素联合起来，生成正则表达式
     * 
     * @param array
     * @return
     */
    @SuppressWarnings("squid:S5361")
    private static String generateRegex(String ... array) {
        if (array == null || array.length == 0) {
            return "";
        }

        StringBuilder builder = new StringBuilder();

        int len = array.length;
        for (int i = 0; i < len; i++) {
            String str = array[i];

            // 替换掉正则表达式中的语义符号
            str = str.replaceAll("\\\\", "\\\\\\\\");// '\'
            str = str.replaceAll("\\|", "\\\\|");// '|'
            str = str.replaceAll("\\)", "\\\\)");// ')'
            str = str.replaceAll("\\(", "\\\\(");// '('
            str = str.replaceAll("\\]", "\\\\]");// ']'
            str = str.replaceAll("\\[", "\\\\[");// '['
            str = str.replaceAll("\\+", "\\\\+");// '+'
            str = str.replaceAll("\\?", "\\\\?");// '?'
            str = str.replaceAll("\\*", "\\\\*");// '*'
            str = str.replaceAll(Matcher.quoteReplacement("$"), Matcher.quoteReplacement("\\$"));// '$'

            builder.append(str);

            // 使用 '|' 连接两组词素
            if (i != len - 1) {
                builder.append('|');
            }
        }

        return builder.toString();
    }

    /**
     * 替换字符串，将编码表中的词素转义成另一组词素。
     * 
     * @param source
     * @param regex
     * @param map
     * @return
     */
    public static String replace(String source, String regex, Map<String, String> map) {
        if (source == null || source.trim().length() == 0) {
            return source;
        }

        if (source.length() > 0) {
            Map<String, String> tmpMap = new HashMap<>();

            // 使用正则表达式检查匹配的文本
            Pattern emoji = Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
            Matcher m = emoji.matcher(source);
            while (m.find()) {
                String key = m.group();
                String value = map.get(key);
                if (value != null) {
                    tmpMap.put(key, value);
                }
            }

            // 逐一替换命中的符号
            if (!tmpMap.isEmpty()) {
                for (Entry<String, String> entry : tmpMap.entrySet()) {
                    String key = entry.getKey();
                    String value = entry.getValue();
                    source = source.replace(key, value);
                }
            }
        }
        return source;
    }

    /**
     * 判断字符串中是否包含正则式中的词素
     * 
     * @param source 字符串
     * @param regex 正则表达式
     * @return true 包含, false 不包含
     */
    public static boolean contains(String source, String regex) {
        boolean result = false;
        if (source != null && source.length() > 0) {
            Pattern p = Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
            Matcher m = p.matcher(source);
            if (m.find()) {
                result = true;
            }
        }
        return result;
    }

    /**
     * 将字符串中所有的表情符号，转换为文字描述。
     * 
     * 例如:
     * 
     * <ul>
     * <li>/::) -&gt; [微笑]</li>
     * <li>/::D -&gt; [呲牙]</li>
     * </ul>
     * @param source
     * @return
     */
    public static String decode(String source) {
        return replace(source, DECODE_REGEX, DECODE_MAP);
    }

    /**
     * 将字符串中所有的表情的文字描述转换为符号。
     * 
     * 例如:
     * 
     * <ul>
     * <li>[微笑] -&gt; /::)</li>
     * <li>[呲牙] -&gt; /::D</li>
     * </ul>
     * @param source
     * @return
     */
    public static String encode(String source) {
        return replace(source, ENCODE_REGEX, ENCODE_MAP);
    }

    /**
     * 判断字符串中是否含有微信表情符号
     * 
     * @param source
     * @return
     */
    public static boolean containsCode(String source) {
        return contains(source, DECODE_REGEX);
    }

    /**
     * 判断字符串中是否含有微信表情的文字描述
     * 
     * @param source
     * @return
     */
    public static boolean containsWord(String source) {
        return contains(source, ENCODE_REGEX);
    }
}