package cn.kinyun.scrm.weixin.sdk.api.applet;

import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.req.*;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.AppletGenerateSchemeResp;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.AppletSessionResp;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.GetUserPhoneResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Objects;

/**
 * @author zhangjing
 * @desc
 * @date 2022/1/26 3:58 下午
 */
@Slf4j
@Component
public class AppletAPI {

    @Autowired
    private RestTemplate restTemplate;

    @Value("${wx.applet.sns.jscode2session.url}")
    private String snsJsCode2Session;

    @Value("${wx.applet.getPhoneNumber.url}")
    private String getPhoneNumberUrl;

    @Value("${wx.applet.sns.plain.jscode2session.url}")
    private String snsPlainJsCode2SessionUrl;

    @Value("${wx.applet.generatescheme.url}")
    private String wxGenerateSchemeUrl;

    /**
     * 开放平台绑定小程序登录,并获取session
     *
     * @param req
     * @return
     */
    public AppletSessionResp code2Session(JsCode2SessionReq req) {
        log.info("code2Session with req={}", req);
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        // 发送请求
        String url = MessageFormat.format(snsJsCode2Session, req.getAppId(), req.getJsCode(), req.getComponentAppId(), req.getComponentAccessToken());
        ResponseEntity<AppletSessionResp> resp = restTemplate.getForEntity(url, AppletSessionResp.class);
        AppletSessionResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    public GetUserPhoneResp getUserPhoneNumber(String accessToken, String code) {
        log.info("getUserPhoneNumber with code={}", code);
        Preconditions.checkArgument(Objects.nonNull(code), "code不能为空");
        Preconditions.checkArgument(Objects.nonNull(accessToken), "accessToken不能为空");

        GetUserPhoneReq req = new GetUserPhoneReq();
        req.setCode(code);

        String url = MessageFormat.format(getPhoneNumberUrl, accessToken);
        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<GetUserPhoneReq> httpEntity = new HttpEntity<>(req, headers);

        ResponseEntity<GetUserPhoneResp> resp = restTemplate.postForEntity(url, httpEntity, GetUserPhoneResp.class);
        GetUserPhoneResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 小程序登录,并获取session
     * @param req
     * @return
     */
    public AppletSessionResp plainCode2Session(PlainJsCode2Session req) {
        log.info("plainCode2Session with req={}", req);
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        // 发送请求
        String url = MessageFormat.format(snsPlainJsCode2SessionUrl, req.getAppId(), req.getJsCode(), req.getSecret());
        ResponseEntity<AppletSessionResp> resp = restTemplate.getForEntity(url, AppletSessionResp.class);
        AppletSessionResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 小程序获取scheme码
     * @param accessToken
     * @param req
     * @return
     */
    public AppletGenerateSchemeResp generateScheme(@NonNull String accessToken, AppletGenerateSchemeReq req) {
        log.info("generateScheme with req={}", req);
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        // 发送请求
        String url = MessageFormat.format(wxGenerateSchemeUrl, accessToken);

        // 构造请求体
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<AppletGenerateSchemeReq> httpEntity = new HttpEntity<>(req, headers);

        ResponseEntity<AppletGenerateSchemeResp> resp = restTemplate.postForEntity(url, httpEntity, AppletGenerateSchemeResp.class);
        AppletGenerateSchemeResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
