/**
 * kuaike.com Inc. Copyright (c) 2014-2019 All Rights Reserved.
 */
package cn.kinyun.scrm.weixin.sdk.api.analysis;

import cn.kinyun.scrm.weixin.sdk.entity.analysis.AnalysisReq;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.article.ArticleSummary;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.article.ArticleTotal;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.article.UserRead;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.article.UserReadHour;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.article.UserShare;
import cn.kinyun.scrm.weixin.sdk.entity.analysis.article.UserShareHour;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;

import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * 微信图文数据分析接口
 * 
 * <p>
 * 在公众号登录授权机制的权限集划分中，图文分析数据接口属于群发与通知权限。
 * </p>
 * 
 * @see <a href=
 *      "https://developers.weixin.qq.com/doc/offiaccount/Analytics/Graphic_Analysis_Data_Interface.html">数据统计-图文分析</a>
 * @title WxArticleAnalysisAPI
 * @author yanmaoyuan
 * @date 2019年9月24日
 * @version 1.0
 */
@Slf4j
@Component
public class WxArticleAnalysisAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * 获取图文群发每日数据 POST
     */
    @Value("${wx.analysis.getarticlesummary}")
    private String wxAnalysisGetArticleSummary;

    /**
     * 获取图文群发总数据 POST
     */
    @Value("${wx.analysis.getarticletotal}")
    private String wxAnalysisGetArticleTotal;

    /**
     * 获取图文统计数据 POST
     */
    @Value("${wx.analysis.getuserread}")
    private String wxAnalysisGetUserRead;

    /**
     * 获取图文统计分时数据 POST
     */
    @Value("${wx.analysis.getuserreadhour}")
    private String wxAnalysisGetUserReadHour;

    /**
     * 获取图文分享转发数据 POST
     */
    @Value("${wx.analysis.getusershare}")
    private String wxAnalysisGetUserShare;

    /**
     * 获取图文分享转发分时数据 POST
     */
    @Value("${wx.analysis.getusersharehour}")
    private String wxAnalysisGetUserShareHour;

    /**
     * 获取图文群发每日数据
     * 
     * <p>
     * 最大时间跨度:1天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public ArticleSummary getArticleSummary(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get article summary with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetArticleSummary, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<ArticleSummary> response = restTemplate.postForEntity(url, request, ArticleSummary.class);

        ArticleSummary result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取图文群发总数据
     * 
     * <p>
     * 最大时间跨度:1天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public ArticleTotal getArticleTotal(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get article total with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetArticleTotal, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<ArticleTotal> response = restTemplate.postForEntity(url, request, ArticleTotal.class);

        ArticleTotal result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取图文统计数据
     * 
     * <p>
     * 最大时间跨度:3天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UserRead getUserRead(@NonNull String accessToken, @NonNull AnalysisReq params) throws WeixinException {
        log.info("get user read with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUserRead, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UserRead> response = restTemplate.postForEntity(url, request, UserRead.class);

        UserRead result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取图文统计分时数据
     * 
     * <p>
     * 最大时间跨度:1天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UserReadHour getUserReadHour(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get user read hour with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUserReadHour, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UserReadHour> response = restTemplate.postForEntity(url, request, UserReadHour.class);

        UserReadHour result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取图文分享转发数据
     * 
     * <p>
     * 最大时间跨度:7天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UserShare getUserShare(@NonNull String accessToken, @NonNull AnalysisReq params) throws WeixinException {
        log.info("get user share with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUserShare, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UserShare> response = restTemplate.postForEntity(url, request, UserShare.class);

        UserShare result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取图文分享转发分时数据
     * 
     * <p>
     * 最大时间跨度:1天
     * </p>
     * 
     * @param accessToken
     * @param params
     * @return
     * @throws WeixinException
     */
    public UserShareHour getUserShareHour(@NonNull String accessToken, @NonNull AnalysisReq params)
        throws WeixinException {
        log.info("get user share hour with params={}", params);

        String url = MessageFormat.format(wxAnalysisGetUserShareHour, accessToken);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<AnalysisReq> request = new HttpEntity<AnalysisReq>(params, headers);
        ResponseEntity<UserShareHour> response = restTemplate.postForEntity(url, request, UserShareHour.class);

        UserShareHour result = response.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}