package cn.kinyun.scrm.weixin.sdk.api.shop;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.shop.dto.SpuResultDto;
import cn.kinyun.scrm.weixin.sdk.entity.shop.req.*;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.SpuAddResp;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.SpuGetResp;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.SpuListResp;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.SpuUpdateResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import cn.kinyun.scrm.weixin.sdk.utils.JsonUtil;
import com.google.common.base.Preconditions;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Objects;

/**
 * @Desc
 * @Author zhangjing
 * @Date 2021/12/31 4:51 下午
 * @Version 1.0
 */
@Slf4j
@Component
public class WxShopSpuAPI {

    @Autowired
    private RestTemplate restTemplate;

    /**
     * SPU接口-添加商品 POST
     */
    @Value("${wx.shop.spu.add}")
    private String shopSpuAddUrl;

    /**
     * SPU接口-删除商品 POST
     */
    @Value("${wx.shop.spu.del}")
    private String shopSpuDelUrl;

    /**
     * SPU接口-获取商品 POST
     */
    @Value("${wx.shop.spu.get}")
    private String shopSpuGetUrl;

    /**
     * SPU接口-获取商品列表 POST
     */
    @Value("${wx.shop.spu.get_list}")
    private String shopSpuGetListUrl;

    /**
     * SPU接口-搜索商品 POST
     */
    @Value("${wx.shop.spu.search}")
    private String shopSpuSearchUrl;

    /**
     * SPU接口-更新商品 POST
     */
    @Value("${wx.shop.spu.update}")
    private String shopSpuUpdateUrl;

    /**
     * SPU接口-上架商品 POST
     */
    @Value("${wx.shop.spu.listing}")
    private String shopSpuListingUrl;

    /**
     * SPU接口-下架商品 POST
     */
    @Value("${wx.shop.spu.delisting}")
    private String shopSpuDeListingUrl;

    /**
     * 添加商品
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SpuAddResp spuAdd(@NonNull String accessToken, @NonNull SpuAddReq req) {
        log.info("spuAdd with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<SpuAddResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        // 构造请求体
        byte[] data = JsonUtil.obj2Str(req).getBytes();
        HttpEntity<?> httpEntity = new HttpEntity<>(data, headers);

        // 发送请求
        String url = MessageFormat.format(shopSpuAddUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SpuAddResp.class);

        SpuAddResp result = resp.getBody();

        log.info("spuAdd with result={}", result);
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取商品
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SpuGetResp spuGet(@NonNull String accessToken, @NonNull SpuGetReq req) {
        log.info("spuGet with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        ResponseEntity<SpuGetResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SpuProductIdReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSpuGetUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SpuGetResp.class);

        SpuGetResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取商品列表
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SpuListResp spuList(@NonNull String accessToken, @NonNull SpuListReq req) {
        log.info("spuList with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        ResponseEntity<SpuListResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SpuListReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSpuGetListUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SpuListResp.class);

        SpuListResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 删除商品
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode spuDel(@NonNull String accessToken, @NonNull SpuProductIdReq req) {
        log.info("spuDel with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SpuProductIdReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSpuDelUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 搜索商品
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SpuResultDto spuSearch(@NonNull String accessToken, @NonNull SpuSearchReq req) {
        log.info("spuSearch with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<SpuResultDto> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SpuSearchReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSpuSearchUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SpuResultDto.class);

        SpuResultDto result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 更新商品
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SpuUpdateResp spuUpdate(@NonNull String accessToken, @NonNull SpuUpdateReq req) {
        log.info("spuUpdate with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<SpuUpdateResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SpuUpdateReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSpuUpdateUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SpuUpdateResp.class);

        SpuUpdateResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 上架商品
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode spuListing(@NonNull String accessToken, @NonNull SpuProductIdReq req) {
        log.info("spuListing with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SpuProductIdReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSpuListingUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 下架商品
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode spuDelisting(@NonNull String accessToken, @NonNull SpuProductIdReq req) {
        log.info("spuDelisting with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SpuProductIdReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSpuDeListingUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
