package cn.kinyun.scrm.weixin.sdk.api.shop;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.shop.req.*;
import cn.kinyun.scrm.weixin.sdk.entity.shop.resp.*;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Objects;

/**
 * @author zhangjing
 * @date 2022/1/4 11:12 上午
 */
@Slf4j
@Component
public class WxShopSkuAPI {
    @Autowired
    private RestTemplate restTemplate;

    /**
     * SKU接口-添加sku POST
     */
    @Value("${wx.shop.sku.add}")
    private String shopSkuAddUrl;

    /**
     * SKU接口-添加sku POST
     */
    @Value("${wx.shop.sku.batch_add}")
    private String shopSkuBatchAddUrl;

    /**
     * SKU接口-获取sku信息 POST
     */
    @Value("${wx.shop.sku.get}")
    private String shopSkuGetUrl;

    /**
     * SKU接口-批量获取sku信息 POST
     */
    @Value("${wx.shop.sku.get_list}")
    private String shopSkuGetListUrl;

    /**
     * SKU接口-删除sku POST
     */
    @Value("${wx.shop.sku.del}")
    private String shopSkuDelUrl;

    /**
     * SKU接口-更新sku POST
     */
    @Value("${wx.shop.sku.update}")
    private String shopSkuUpdateUrl;

    /**
     * SKU接口-更新sku价格 POST
     */
    @Value("${wx.shop.sku.update_price}")
    private String shopSkuUpdatePriceUrl;

    /**
     * SKU接口-更新库存 POST
     */
    @Value("${wx.shop.stock.update}")
    private String shopStockUpdateUrl;

    /**
     * SKU接口-获取库存 POST
     */
    @Value("${wx.shop.stock.get}")
    private String shopStockGetUrl;

    /**
     * 添加sku
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SkuAddResp skuAdd(@NonNull String accessToken, @NonNull SkuAddReq req) {
        log.info("skuAdd with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<SkuAddResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SkuAddReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSkuAddUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SkuAddResp.class);

        SkuAddResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 批量添加sku
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SkuBatchAddResp skuBatchAdd(@NonNull String accessToken, @NonNull SkuBatchAddReq req) {
        log.info("skuBatchAdd with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<SkuBatchAddResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SkuBatchAddReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSkuBatchAddUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SkuBatchAddResp.class);

        SkuBatchAddResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 删除sku
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode skuDel(@NonNull String accessToken, @NonNull SkuIdReq req) {
        log.info("skuDel with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SkuIdReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSkuDelUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取sku信息
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SkuGetResp skuGet(@NonNull String accessToken, @NonNull SkuGetReq req) {
        log.info("skuGet with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<SkuGetResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SkuGetReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSkuGetUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SkuGetResp.class);

        SkuGetResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 批量获取sku信息
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SkuGetListResp skuGetBatch(@NonNull String accessToken, @NonNull SkuGetListReq req) {
        log.info("skuGetBatch with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<SkuGetListResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SkuGetListReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSkuGetListUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SkuGetListResp.class);

        SkuGetListResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 更新SKU
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SkuUpdateResp skuUpdate(@NonNull String accessToken, @NonNull SkuUpdateReq req) {
        log.info("skuUpdate with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<SkuUpdateResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SkuUpdateReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSkuUpdateUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SkuUpdateResp.class);

        SkuUpdateResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 更新SKU价格
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SkuUpdatePriceResp skuUpdatePrice(@NonNull String accessToken, @NonNull SkuUpdatePriceReq req) {
        log.info("skuUpdatePrice with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<SkuUpdatePriceResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SkuUpdatePriceReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopSkuUpdatePriceUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SkuUpdatePriceResp.class);

        SkuUpdatePriceResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 更新库存
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SkuUpdateStockResp skuUpdateStock(@NonNull String accessToken, @NonNull SkuUpdateStockReq req) {
        log.info("skuUpdateStock with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<SkuUpdateStockResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SkuUpdateStockReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopStockUpdateUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SkuUpdateStockResp.class);

        SkuUpdateStockResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 获取库存
     *
     * @param accessToken
     * @param req
     * @return
     */
    public SkuGetStockResp skuGetStock(@NonNull String accessToken, @NonNull SkuGetStockReq req) {
        log.info("skuGetStock with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        ResponseEntity<SkuGetStockResp> resp = null;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<SkuGetStockReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(shopStockGetUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, SkuGetStockResp.class);

        SkuGetStockResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
