package cn.kinyun.scrm.weixin.sdk.api.applet;

import cn.kinyun.scrm.weixin.sdk.entity.ErrorCode;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.req.ApplyPluginReq;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.req.PluginListReq;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.req.UnbindPluginReq;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.req.UpdatePluginReq;
import cn.kinyun.scrm.weixin.sdk.entity.miniprogram.resp.AppletPluginListResp;
import cn.kinyun.scrm.weixin.sdk.exception.WeixinException;
import com.google.common.base.Preconditions;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.Objects;

/**
 * @author zhangjing
 * @desc
 * @date 2022/1/20 4:08 下午
 */
@Slf4j
@Component
public class AppletPluginAPI {

    @Autowired
    private RestTemplate restTemplate;

    @Value("${wx.applet.plugin.url}")
    private String appletPluginUrl;

    /**
     * 申请小程序插件
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode applyPlugin(@NonNull String accessToken, @NonNull ApplyPluginReq req) {
        log.info("applyPlugin with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");
        req.validate();

        ResponseEntity<ErrorCode> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<ApplyPluginReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(appletPluginUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 查询当前已添加的小程序插件
     *
     * @param accessToken
     * @return
     */
    public AppletPluginListResp getPluginList(@NonNull String accessToken) {
        log.info("getPluginList");

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");

        ResponseEntity<AppletPluginListResp> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        PluginListReq req = new PluginListReq();
        req.setAction("list");
        // 构造请求体
        HttpEntity<PluginListReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(appletPluginUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, AppletPluginListResp.class);

        AppletPluginListResp result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 解绑小程序插件
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode unbindPlugin(@NonNull String accessToken, @NonNull UnbindPluginReq req) {
        log.info("unbindPlugin with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<UnbindPluginReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(appletPluginUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }

    /**
     * 快速更新小程序插件版本号
     *
     * @param accessToken
     * @param req
     * @return
     */
    public ErrorCode updatePlugin(@NonNull String accessToken, @NonNull UpdatePluginReq req) {
        log.info("updatePlugin with req={}", req);

        Preconditions.checkArgument(StringUtils.isNoneBlank(accessToken), "accessToken不能为空");
        Preconditions.checkArgument(Objects.nonNull(req), "req不能为空");

        ResponseEntity<ErrorCode> resp;

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        // 构造请求体
        HttpEntity<UpdatePluginReq> httpEntity = new HttpEntity<>(req, headers);

        // 发送请求
        String url = MessageFormat.format(appletPluginUrl, accessToken);
        resp = restTemplate.postForEntity(url, httpEntity, ErrorCode.class);

        ErrorCode result = resp.getBody();
        WeixinException.isSuccess(result);
        return result;
    }
}
